/* Using jet lib: jet_tree. general lib */
#include "lorenz4.h"

/***********************************************************************
 *
 * Code generated by the TAYLOR translator.
 */

#define _N_DIM_ 3
#define _J_DIM_ 3


#define xInitUpJet InitUpJet
#define xInitJet InitJet
#define xNormJet NormJet
#define xMultiplyFloatJetA MultiplyFloatJetA
#define xAddJetJetA AddJetJetA 
#define xAssignJetToJet AssignJetToJet
#define xAssignJetToFloat AssignJetToFloat





/*
next line defines the largest power of 2 such that 2^(LEXP2) and
2^(-LEXP2) do not overflow/underflow the double arithmetic of your
computer.
*/
#define LEXP2 1023 

#define DEBUG_LEVEL 0 /* to print some internal information */

int taylor_step_lorenz4_eq(MY_FLOAT *ti,
                 MY_FLOAT *x,
                 int      dir,
                 int      step_ctl,
                 double   log10abserr,
                 double   log10relerr,
                 MY_FLOAT *endtime,
                 MY_FLOAT *ht,
                 int      *order,
                 MY_JET   *jetInOut)
/*
 * single integration step with taylor method. the parameters are:
 *
 * ti: on input:  time of the initial condition
 *     on output: new time
 *
 * x:  on input:  initial condition
 *     on output: new condition, corresponding to the (output) time ti
 *
 * dir: flag to integrate forward or backwards.
 *     1: forward
 *    -1: backwards
 *     WARNING: this flag is ignored if step_ctl (see below) is set to 0.
 *
 * step_ctl: flag for the step size control. the possible values are:
 *     0: no step size control, so the step and order are provided by
 *        the user. the parameter ht is used as step, and the parameter
 *        order (see below) is used as the order.
 *     1: standard stepsize control. it uses an approximation to the
 *        optimal order and to the radius of convergence of the series
 *        to approximate the 'optimal' step size. It tries to keep
 *        either the absolute or the relative errors below the given
 *        values. See the paper for more details.
 *     2: as 1, but adding an extra condition on the stepsize h: the
 *        terms of the series --after being multiplied by the suitable
 *        power of h-- cannot grow.
 *    -1: user defined stepsize control. the code has to be included
 *        in the routine compute_timestep_user_defined (you can find
 *        this routine below). The user should also provide code for
 *        the selection of degree (see the function
 *        compute_order_user_defined below).
 *
 * log10abserr: decimal log of the absolute accuracy. the routine
 *     tries to keep either the absolute or the relative errors below
 *     the given thresholds.
 *
 * log10relerr: decimal log of the relative accuracy. the routine
 *     tries to keep either the absolute or the relative errors below
 *     the given thresholds.
 *
 * endtime: if NULL, it is ignored. if step_ctl (see above) is set
 *     to 0, it is also ignored. otherwise, if the step size is too
 *     large, it is reduced so that the new time ti is exactly endtime
 *     (in that case, the function returns 1).
 *
 * ht: on input:  ignored/used as a time step (see parameter step_ctl)
 *     on output: time step used
 *
 * order: degree of the taylor expansion.
 *        input: this parameter is only used if step_ctl is 0,
 *               or if you add the code for the case step_ctl=3.
 *               its possible values are:
 *               < 2: the program will select degree 2 (if step_ctl is 0).
 *               >=2: the program will use this degree (if step_ctl is 0).
 *        ouput: degree used.
 *
 * jetInOut: on input: the value of all declared jet variables
 *           on output: new value of the jet variable, corresponding to the new time
 *
 * return value:
 * 0: ok.
 * 1: ok, and ti=endtime.  */
{
  MY_FLOAT **taylor_coefficients_lorenz4_eq(MY_FLOAT, MY_FLOAT*, int);
  MY_FLOAT **taylor_coefficients_lorenz4_eqA(MY_FLOAT, MY_FLOAT*, int, int);
  int compute_order_1_lorenz4_eq(double, double, double, int*);
  int comp_order_other_lorenz4_eq(double, double, double);
  double compute_stepsize_1_lorenz4_eq(MY_FLOAT**, MY_JET**, int, double, int);
  double compute_stepsize_2_lorenz4_eq(MY_FLOAT**, MY_JET**, int, double, int);
  double comp_stepsize_other_lorenz4_eq(MY_FLOAT**, MY_JET**, int, int, double, double, double);

  static MY_FLOAT **s,h,mtmp;
  static MY_JET jtmp,jtmp1, **jetJetOut;
  double xi,xnorm,dh;
  int i,j,k,nt,flag_endtime,flag_err;
  static int init=0;
#pragma omp threadprivate(s,h,mtmp,init)
#pragma omp threadprivate(jtmp,jtmp1,jetJetOut)

  if (init == 0) /* initialization of MY_FLOAT and MY_JET variables */
    {
      init=1;
      InitMyFloat(h);
      InitMyFloat(mtmp);
      xInitJet(jtmp);
      xInitJet(jtmp1);
      xInitUpJet();
    }
/*
  sup norm of the initial condition (jet included)
*/
  xnorm=0;
  if (step_ctl != 0)
    {
#if _J_DIM_ != 0
    for (i=0; i<_J_DIM_; i++)
      {
        xNormJet(mtmp,jetInOut[i]);
        MyFloatToDouble(xi,mtmp);
        xi=fabs(xi);
        if (xi > xnorm) xnorm=xi;
      }
#endif
      for (i=_J_DIM_; i<_N_DIM_; i++)
      {
	MyFloatToDouble(xi,x[i]);
	xi=fabs(xi);
	if (xi > xnorm) xnorm=xi;
      }
    }
/*
  we determine the degree of the taylor expansion.
  this value will be stored in the variable nt.
  the flag flag_err returns a value indicating if we are using an
  absolute error estimate (value 1) or a relative one (value 2).
*/
  switch(step_ctl)
    {
    case 0: /* no step size control, fixed degree; both given by the user */
      nt=(*order<2)? 2: *order; /* 2 is the minimum order allowed */
      break;
    case 1:
      nt=compute_order_1_lorenz4_eq(xnorm,log10abserr,log10relerr,&flag_err);
      break;
    case 2:
      nt=compute_order_1_lorenz4_eq(xnorm,log10abserr,log10relerr,&flag_err);
      break;
    case -1:
      nt=comp_order_other_lorenz4_eq(xnorm,log10abserr,log10relerr);
      break;
    default:
      fprintf(stderr, "taylor_step: undefined step size control.\n");
      fprintf(stderr, "you must choose an existing step size control\n");
      fprintf(stderr, "method or supply a step size control procedure.\n");
      exit(0);
    }
  *order=nt;
/*
  computation of the jet of derivatives up to order nt
*/
  if(step_ctl != 0) {
    s=taylor_coefficients_lorenz4_eq_A(*ti,x,nt,1, jetInOut, &jetJetOut);
  } else {
    s=taylor_coefficients_lorenz4_eq_A(*ti,x,nt,0, jetInOut, &jetJetOut); 
 }

/*
  selection of the routine to compute the time step. the value
  step_ctl=3 has been reserved for the user, in case she/he wants to
  code a different method.
*/
  switch(step_ctl)
    {
    case 0: /* no step size control (fixed step size, given by the user) */
      AssignMyFloat(h,*ht);
      break;
    case 1:
      dh=compute_stepsize_1_lorenz4_eq(s,jetJetOut,nt,xnorm,flag_err);
      MakeMyFloatA(h, dh);
      break;
    case 2:
      dh=compute_stepsize_2_lorenz4_eq(s,jetJetOut,nt,xnorm,flag_err);
      MakeMyFloatA(h, dh);
      break;
    case -1:
      dh=comp_stepsize_other_lorenz4_eq(s,jetJetOut,_N_DIM_,nt,xnorm,log10abserr,log10relerr);
      MakeMyFloatA(h, dh);
      break;
    default:
      fprintf(stderr, "taylor_step: undefined step size control.\n");
      fprintf(stderr, "You must choose an existing step size control\n");
      fprintf(stderr, "method or supply a step size control procedure.\n");
      exit(0);
    }
/*
  if step_ctl != 0, we adjust the sign of the computed stepsize.
*/
  flag_endtime=0;
  if (step_ctl != 0)
    {
      if (dir == -1) { NegateMyFloatA(mtmp,h); AssignMyFloat(h, mtmp);} 
/*
      we compare *ti+h with endtime. we modify h if necessary.
*/
      if (endtime != NULL)
        {
          AddMyFloatA(mtmp,h,*ti);
          if (dir == 1) /* time goes forward */
            {
              if (MyFloatA_GE_B(mtmp,*endtime))
                {
                  SubtractMyFloatA(h,*endtime,*ti);
                  flag_endtime=1;
                }
            }
            else /* time goes backwards */
            {
              if (MyFloatA_GE_B(*endtime,mtmp))
                {
                  SubtractMyFloatA(h,*endtime,*ti);
                  flag_endtime=1;
                }
            }
        }
    }
/*
  next lines are the summation of the taylor series (horner's method)
*/
  j=nt-1;
#if _J_DIM_ != 0
  for(i=0; i<_J_DIM_; i++)
  {
    xAssignJetToJet(jetInOut[i],jetJetOut[i][nt]);
    for(k=j; k>=0; k--)
    {
      xMultiplyFloatJetA(jtmp, h, jetInOut[i]);
      xAddJetJetA(jetInOut[i], jtmp, jetJetOut[i][k]);
    }
    xAssignJetToFloat(x[i],jetInOut[i]);
  }
#endif
  for(i=_J_DIM_; i<_N_DIM_; i++)
  {
    AssignMyFloat(x[i],s[i][nt]);
    for(k=j; k>=0; k--)
    {
      MultiplyMyFloatA(mtmp, h, x[i]);
      AddMyFloatA(x[i], mtmp, s[i][k]);
    }
  }
/*
  finally, we set the values of the parameters *ht and *ti.
*/
  AssignMyFloat(*ht,h);
  if (flag_endtime == 0)
    {
      AssignMyFloat(mtmp, *ti);
      AddMyFloatA(*ti, mtmp, h);
    }
    else
    {
      AssignMyFloat(*ti,*endtime);
    }
  return(flag_endtime);
}
int compute_order_1_lorenz4_eq(double xnorm, double log10abserr, double log10relerr, int* flag_err)
/*
 * this is to determine the 'optimal' degree of the taylor expansion.
 *
 * parameters:
 * xnorm: norm of the initial condition
 * log10abserr: base-10 log of the absolute error required
 * log10relerr: base-10 log of the relative error required
 * flag_err:    flag. returns 1 if absolute error is used
 *                    returns 2 if relative error is used
 *
 * returned value: 'optimal' degree.
*/
{
  double log10eps,tmp;
  int nt;

  log10eps=log10abserr;
  *flag_err=1;
  if (xnorm != (double)0.0)
    {
      tmp=log10relerr+log10(xnorm);
      if (tmp > log10abserr) {log10eps=log10relerr; *flag_err=2;}
    }
/*
  we use 1.16 for the value 0.5*log(10)=1.151292546497...
*/
  nt=(int)(1.5-1.16*log10eps);
  if (nt < 2) nt=2; /* this is the minimum order accepted */

#if DEBUG_LEVEL > 0
      fprintf(stderr, "taylor_step: order is %d\n",nt);
#endif

  return(nt);
}
double compute_stepsize_1_lorenz4_eq(MY_FLOAT **s, MY_JET **jet, int nt, double xnorm, int flag_err)
/*
 * it looks for a step size for an expansion up to order nt. this
 * function requires that nt is the value computed by
 * compute_order_1_
 */
{
  double double_log_MyFloat_lorenz4_eq(MY_FLOAT x);
  static MY_FLOAT z,v1,v2;
  static MY_FLOAT of,uf;
  double lnv1,lnv2,r,lnro1,lnro2,lnro;
  int i;
  static int init=0;
#pragma omp threadprivate(z,v1,v2,of,uf,init)

  if (init == 0)
    {
      init=1;
      InitMyFloat(z);
      InitMyFloat(v1);
      InitMyFloat(v2);
      InitMyFloat(of);
      InitMyFloat(uf);

      r=pow((double)2,(double)LEXP2);
      MakeMyFloatA(of,r);
      r=pow((double)2,(double)(-LEXP2));
      MakeMyFloatA(uf,r);
    }
/*
  we compute the sup norm of the last two coefficients of the taylor
  series, and we store them into v1 and v2.
*/
  MakeMyFloatA(v1,0);
  MakeMyFloatA(v2,0);
#if _J_DIM_ != 0
  for (i=0; i<_J_DIM_; i++)
  {
    xNormJet(z,jet[i][nt-1]);
    if (MyFloatA_GT_B(z,v1)) AssignMyFloat(v1,z);
    xNormJet(z,jet[i][nt]);
    if (MyFloatA_GT_B(z,v2)) AssignMyFloat(v2,z);
  }
#endif
  for(i=_J_DIM_; i<_N_DIM_; i++)
  {
    fabsMyFloatA(z,s[i][nt-1]);
    if (MyFloatA_GT_B(z,v1)) AssignMyFloat(v1,z);
    fabsMyFloatA(z,s[i][nt]);
    if (MyFloatA_GT_B(z,v2)) AssignMyFloat(v2,z);
  }
/*
  computation of the step size. we need the logs of v1 and v2, in
  double precision (there is no need for extended precision). the idea
  is to assign these variables to double variables and then to use the
  standard log function. before doing that, we have to be sure that v1
  can be assigned to a double without under or overflow. for this
  reason we will check for this condition and, if it fails, we will
  call an specific function for this case.
*/
  if (MyFloatA_LE_B(v1,of) && MyFloatA_GE_B(v1,uf))
    {
      MyFloatToDouble(r,v1);
      lnv1=log(r);
    }
    else
    {
      lnv1=double_log_MyFloat_lorenz4_eq(v1);
    }
  if (MyFloatA_LE_B(v2,of) && MyFloatA_GE_B(v2,uf))
    {
      MyFloatToDouble(r,v2);
      lnv2=log(r);
    }
    else
    {
      lnv2=double_log_MyFloat_lorenz4_eq(v2);
    }
/*
  if flag_err == 2, this means that we are using a relative error control.
*/
  if (flag_err == 2)
    {
      r = -log10(xnorm);
      lnv1 += r;
      lnv2 += r;
    }
 lnro1= -lnv1/(nt-1);
 lnro2= -lnv2/nt;
 lnro=(lnro1 < lnro2)? lnro1: lnro2;

 r=exp(lnro-2-0.7/(nt-1)); /* exp(-0.7/(nt-1)) is a security factor */
  return(r);
}
double compute_stepsize_2_lorenz4_eq(MY_FLOAT **s, MY_JET **jet, int nt, double xnorm, int flag_err)
/*
 * it looks for a step size for an expansion up to order nt. this
 * function requires that nt is the value computed by
 * compute_order_1_. it also tries to reduce cancellations of big
 * terms in the summation of the taylor series.
 */
{
  double compute_stepsize_1_lorenz4_eq(MY_FLOAT**, MY_JET**, int, double, int);
  static MY_FLOAT h,hj,r,z,a,normj;
  double c,rtmp,dh;
  int i,j;
  static int init=0;
#pragma omp threadprivate(h,hj,r,z,a,normj,init)

  if (init == 0)
    {
      init=1;
      InitMyFloat(h); 
      InitMyFloat(hj);
      InitMyFloat(r);
      InitMyFloat(z);
      InitMyFloat(a);
      InitMyFloat(normj);
    }
/*
  we compute the step size according to the first algorithm
*/
  dh=compute_stepsize_1_lorenz4_eq(s,jet,nt,xnorm,flag_err);
  MakeMyFloatA(h,dh);
/*
  next lines select a value (z), that will be used to control the size
  of the terms of the Taylor series.
*/
  if (flag_err == 1) {
     MakeMyFloatA(z, 1.0);
  } else if (flag_err == 2) {
    MakeMyFloatA(z,xnorm);
  } else
    {
      printf("compute_stepsize_2 internal error. flag_err: %d\n",flag_err);
      exit(1);
    }
/*
  next loop checks if the sup norm of the terms in the Taylor series are
  lower than z. if a term is greater than z, the step size h is reduced.
*/
  MakeMyFloatA(hj,(double)1.0);

  for(j=1; j<=nt; j++)
  {
    MultiplyMyFloatA(r,h,hj);
    AssignMyFloat(hj,r);

    MakeMyFloatC(normj,"0", (double)0);
#if _J_DIM_ != 0
    for (i=0; i<_J_DIM_; i++)
    {
      xNormJet(a,jet[i][j]);
      if (MyFloatA_GT_B(a,normj)) AssignMyFloat(normj,a);
    }
#endif
    for (i=_J_DIM_; i<_N_DIM_; i++)
    {
      fabsMyFloatA(a,s[i][j]);
      if (MyFloatA_GT_B(a,normj)) AssignMyFloat(normj,a);
    }

    MultiplyMyFloatA(r,normj,hj);
    if (MyFloatA_LE_B(r,z)) continue;
/*
    we reduce h (and hj)
*/
    DivideMyFloatA(hj,z,normj);

    DivideMyFloatA(a,r,z);
    MyFloatToDouble(c,a);
    c=pow(c,(double)1.e0/(double)j);
    MakeMyFloatA(a,c);
    DivideMyFloatA(r,h,a);
    AssignMyFloat(h,r);

#if DEBUG_LEVEL > 1
    fprintf(stderr, "order %2d. reducing h from %14.6e to %14.6e\n",j,c*h,h);
#endif
  }

  MyFloatToDouble(rtmp,h);
  return(rtmp);
}

double double_log_MyFloat_lorenz4_eq(MY_FLOAT x)
/*
 * natural log, in double precision, of a MY_FLOAT positive number.
 */
{
  static MY_FLOAT a,tmp;
  static MY_FLOAT z,of,uf;
  double b,lx;
  int k;
  static int init=0;
#pragma omp threadprivate(a,tmp,z,of,uf,init)

  if (init == 0)
    {
      init=1;
      InitMyFloat(a);
      InitMyFloat(z);
      InitMyFloat(of);
      InitMyFloat(uf);
      InitMyFloat(tmp);

      b=0;
      MakeMyFloatA(z,b);
      b=pow((double)2,(double)LEXP2);
      MakeMyFloatA(of,b);
      b=pow((double)2,(double)(-LEXP2));
      MakeMyFloatA(uf,b);
    }

  if (MyFloatA_EQ_B(x,z))
    {
      puts("double_log_MyFloat error: zero argument");
      puts("(this is because one of the last two terms of your taylor");
      puts(" expansion is exactly zero)");
      exit(1);
    }

  AssignMyFloat(a,x);

  k=0;
  while(MyFloatA_LT_B(a,uf))
  {
    ++k;
    if(k>3000){fprintf(stderr,"double_log_MyFloat overflow: %d\n", k); exit(1);}
    MultiplyMyFloatA(tmp,a,of);
    AssignMyFloat(a,tmp);
  }
  while(MyFloatA_GT_B(a,of))
  {
    --k;
    if(k<-3000){fprintf(stderr,"double_log_MyFloat underflow: %d\n", k); exit(1);}
    MultiplyMyFloatA(tmp,a,uf);
    AssignMyFloat(a,tmp);
  }

  MyFloatToDouble(b,a);
/*
  lx stands for log(x)
*/
  lx=log(b)-(LEXP2*0.69314718055994530942)*k;

  return(lx);
}


int comp_order_other_lorenz4_eq(double lnxnorm, double log10abserr, double log10relerr){
  puts("---");
  puts("compute_order_user_defined:");
  puts("you have to code this routine");
  puts("or select a different value for the step_ctl parameter");
  puts("---");
  exit(1);

  return(0);
}
double comp_stepsize_other_lorenz4_eq(MY_FLOAT **s, MY_JET **jet, int nd, int nt, double xnorm, double log10abserr, double log10relerr) {

  puts("---");
  puts("compute_timestep_user_defined:");
  puts("you have to code this routine");
  puts("or select a different value for the step_ctl parameter");
  puts("---");
  exit(1);
  return((double)0.00001);
}
/* MACROS TO LINK MY_FLOAT WITH MY_JET COEFFICIENTS */
#ifndef myscal_t
#define myscal_t          MY_FLOAT

#define myscal_init       InitMyFloat
#define myscal_clean      ClearMyFloat

/* assign */
#define myscal_set        AssignMyFloat
#define myscal_set_d      MakeMyFloatB
#define myscal_set_si     MakeMyFloatB
#define myscal_set_zero(r) MakeMyFloatC(r,"0",0)

/* elemental operations */
#define myscal_add2       AddMyFloatA
#define myscal_sub2       SubtractMyFloatA
#define myscal_mul2       MultiplyMyFloatA
#define myscal_div2       DivideMyFloatA

#define myscal_add2_d     AddMyFloatD
#define myscal_sub2_d     SubtractMyFloatD
#define myscal_mul2_d     MultiplyMyFloatD
#define myscal_div2_d     DivideMyFloatD

#define myscal_add2_si    AddMyFloatSI
#define myscal_sub2_si    SubtractMyFloatSI
#define myscal_si_sub2    SubtractSIMyFloat
#define myscal_mul2_si    MultiplyMyFloatSI
#define myscal_div2_si    DivideMyFloatSI

#define myscal_neg        NegateMyFloatA

/* built-in functions */
#define myscal_set_sqrt   sqrtMyFloatA
#define myscal_set_pow    ExponentiateMyFloatA
#define myscal_set_pow_si ExponentiateMyFloatIA

#define myscal_set_sin    sinMyFloatA
#define myscal_set_cos    cosMyFloatA
#define myscal_set_tan    tanMyFloatA
#define myscal_set_atan   atanMyFloatA
#define myscal_set_exp    expMyFloatA
#define myscal_set_log    logMyFloatA
#define myscal_set_sinh   sinhMyFloatA
#define myscal_set_cosh   coshMyFloatA
#define myscal_set_tanh   tanhMyFloatA
#define myscal_set_log10  log10MyFloatA
#define myscal_set_fabs   fabsMyFloatA
#define myscal_to_si      MyFloatToInt
#define myscal_to_d       MyFloatToDouble

/* boolean operations */
#define myscal_ge  MyFloatA_GE_B
#define myscal_gt  MyFloatA_GT_B
#define myscal_le  MyFloatA_LE_B
#define myscal_lt  MyFloatA_LT_B
#define myscal_eq  MyFloatA_EQ_B
#define myscal_neq MyFloatA_NEQ_B

/* output format */
#define myscal_fprintf OutputMyFloat3
#endif /* END MY_SCAL */


/* CODE jet_tree_t_lorenz4_eq */
#include <stdlib.h>
#include <stdio.h>
#include <math.h>

/*table for 3 symbols and degree 5*/
static int num_coefs_homogeneous[] = {\
1, 0, 0, 0, 0, 0, \
1, 1, 1, 1, 1, 1, \
1, 2, 3, 4, 5, 6, \
1, 3, 6, 10, 15, 21, \
};

#define my_scal_tmp jet_tree_saux_lorenz4_eq
#define my_jet_tmp jet_tree_jaux_lorenz4_eq
static myscal_t my_scal_tmp;
static jet_tree_t_lorenz4_eq my_jet_tmp=NULL;
#pragma omp threadprivate(my_scal_tmp,my_jet_tmp)

#ifndef _NUMBER_OF_SYMBOLS_
#define _NUMBER_OF_SYMBOLS_ 1
#endif
#ifndef _DEGREE_OF_JET_VARS_
#define _DEGREE_OF_JET_VARS_ 0
#endif
#define max_nsymb _NUMBER_OF_SYMBOLS_ 
#define max_deg _DEGREE_OF_JET_VARS_ 
/*static int max_nsymb=_NUMBER_OF_SYMBOLS_,max_deg=_DEGREE_OF_JET_VARS_;*/
static int nsymb=0,deg=0;
#pragma omp threadprivate(nsymb,deg)

#define myscal_axpy(y,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_add2(y,y,my_scal_tmp);}
#define myscal_sixpy(y,a,x) {\
  myscal_mul2_si(my_scal_tmp,x,a);\
  myscal_add2(y,y,my_scal_tmp);}

#define myscal_axpy_scal(y,s,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_mul2(my_scal_tmp,my_scal_tmp,s);\
  myscal_add2(y,y,my_scal_tmp);}
#define myscal_axpy_si(y,s,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_mul2_si(my_scal_tmp,my_scal_tmp,s);\
  myscal_add2(y,y,my_scal_tmp);}


#define myscal_axmy(y,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_sub2(y,y,my_scal_tmp);}
#define myscal_sixmy(y,a,x) {\
  myscal_mul2_si(my_scal_tmp,x,a);\
  myscal_sub2(y,y,my_scal_tmp);}

#define myscal_axmy_scal(y,s,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_mul2(my_scal_tmp,my_scal_tmp,s);\
  myscal_sub2(y,y,my_scal_tmp);}
#define myscal_axmy_si(y,s,a,x) {\
  myscal_mul2(my_scal_tmp,a,x);\
  myscal_mul2_si(my_scal_tmp,my_scal_tmp,s);\
  myscal_sub2(y,y,my_scal_tmp);}

/* CODE jet_tree_t_lorenz4_eq_node */


#define jet_tree_nch_lorenz4_eq_node(n,k) num_coefs_homogeneous[n*(max_deg+1)+k]

size_t jet_tree_create_tree_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, myscal_t *coef) {
  static myscal_t *pcoef=NULL;
#pragma omp threadprivate(pcoef)
  int n, k, m;
  size_t mem=0;
  
  if (coef) pcoef = coef;
  n = h->nsymb;
  k = h->deg;
  h->coef = pcoef;
  
  if (n==1) { ++pcoef; h->term=NULL; }
  else if (k == 0) { ++pcoef; h->term=NULL; }
  else {
    mem=(k+1)*sizeof(*h->term);
    h->term = (__typeof__(h->term)) malloc(mem);
    if (h->term==NULL) {
      fprintf(stderr,"%s:%d not enough memory to allocate\n",__FILE__,__LINE__);
      fflush(stderr); exit(2);
    }
    
    for (m = 0; m <= k; ++m) {
      (h->term[m]).nsymb = n-1;
      (h->term[m]).deg = k-m;
      mem+= jet_tree_create_tree_lorenz4_eq_node(h->term + m, NULL);
    }
  }
  return mem;
}

size_t jet_tree_init_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, int nsymb, int deg, myscal_t *coef)
{
  // assert(h);
  h->nsymb = nsymb;
  h->deg = deg;
  h->coef = coef;
  h->term = NULL;
  return jet_tree_create_tree_lorenz4_eq_node(h,coef);
  size_t mem = 0;
  if (nsymb > 2 && deg > 0)
    {
      __typeof__(h->deg) i;

      mem = (deg + 1) * sizeof(*h->term);
      h->term = (__typeof__(h->term)) malloc(mem); // assert(h->term);
      --nsymb;
//      for (i = 0; i <= deg; ++i)
      for (i = deg; i > 0; --i)
        {
          mem  += jet_tree_init_lorenz4_eq_node(h->term + (deg-i), nsymb, i, coef);
          coef += jet_tree_nch_lorenz4_eq_node(nsymb, i);
        }
      mem += jet_tree_init_lorenz4_eq_node(h->term + deg, nsymb, i, coef);
    }
  return mem;
}

void jet_tree_clean_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h)
{
  if (h->term==NULL) return;

  __typeof__(h->term) ht, hf;
  for (ht = h->term, hf = ht + h->deg;
       ht < hf;
       jet_tree_clean_lorenz4_eq_node(ht), ++ht){}
  jet_tree_clean_lorenz4_eq_node(ht);

  free(h->term); h->term = NULL;
}

void jet_tree_set_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a)
{
  // assert(h && a); // assert(h->nsymb == a->nsymb); // assert(h->deg == h->deg);
  if (h == a || h->coef == a->coef) return;

  __typeof__(h->coef) hc, ac, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg), ac = a->coef;
       hc < hf;
       ++hc, ++ac){myscal_set((*hc),(*ac));}
}

void jet_tree_set_scal_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, myscal_t scal)
{
  __typeof__(h->coef) hc, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       ++hc){myscal_set((*hc),scal);}
}

void jet_tree_set_si_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, int d)
{
  __typeof__(h->coef) hc, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       ++hc){myscal_set_si((*hc),d);}
}

void jet_tree_set_d_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, double d)
{
  __typeof__(h->coef) hc, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       ++hc){myscal_set_d((*hc),d);}
}

#define jet_tree_set_zero_lorenz4_eq_node(h,hc,hf)\
{\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg);\
       hc < hf;\
       ++hc){myscal_set_zero(*hc);}\
}\

void jet_tree_nrminf_lorenz4_eq_node(myscal_t nrm[1], jet_tree_ptr_lorenz4_eq_node h)
{
  __typeof__(h->coef) hc, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       ++hc)
   {
      myscal_set_fabs(my_scal_tmp,(*hc));
      if (myscal_lt(*nrm,my_scal_tmp))
      {myscal_set(*nrm,my_scal_tmp);}
   }
}

void jet_tree_nrm2_lorenz4_eq_node(myscal_t nrm[1], jet_tree_ptr_lorenz4_eq_node h)
{
  __typeof__(h->coef) hc, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       ++hc)
   {
      myscal_mul2(my_scal_tmp,(*hc),(*hc));
      myscal_add2(*nrm,*nrm,my_scal_tmp);
   }
}

static void jet_tree_neg_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a)
{
  // assert(h && a); // assert(h->nsymb == a->nsymb); // assert(h->deg == h->deg);
  // assert(h->term && a->term);
//  if (h == a || h->term == a->term) return;

  __typeof__(h->coef) hc, ac, hf;
  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg), ac = a->coef;
       hc < hf;
       ++hc, ++ac){myscal_neg(*hc,*ac);}
}

/* y <- y - a*x */
#define jet_tree_axmy_lorenz4_eq_node(y,a,x,yc,yf,xc) {\
  for (yc = y->coef, yf = y->coef + jet_tree_nch_lorenz4_eq_node(y->nsymb,y->deg), xc = x->coef;\
       yc < yf;\
       ++yc, ++xc){myscal_axmy((*yc),a,(*xc));}\
}

/* y <- y + a*x */
#define jet_tree_axpy_lorenz4_eq_node(y,a,x,yc,yf,xc) {\
  for (yc = y->coef, yf = y->coef + jet_tree_nch_lorenz4_eq_node(y->nsymb,y->deg), xc = x->coef;\
       yc < yf;\
       ++yc, ++xc){myscal_axpy((*yc),a,(*xc));}\
}

/* y <- y + s*a*x */
#define jet_tree_axpy_scal_lorenz4_eq_node(y,s,a,x,yc,yf,xc) {\
  for (yc = y->coef, yf = y->coef + jet_tree_nch_lorenz4_eq_node(y->nsymb,y->deg), xc = x->coef;\
       yc < yf;\
       ++yc, ++xc){myscal_axpy_scal((*yc),s,a,(*xc));}\
}
#define jet_tree_axpy_d_lorenz4_eq_node(y,s,a,x,yc,yf,xc) {\
  for (yc = y->coef, yf = y->coef + jet_tree_nch_lorenz4_eq_node(y->nsymb,y->deg), xc = x->coef;\
       yc < yf;\
       ++yc, ++xc){myscal_axpy_d((*yc),s,a,(*xc));}\
}
#define jet_tree_axpy_si_lorenz4_eq_node(y,s,a,x,yc,yf,xc) {\
  for (yc = y->coef, yf = y->coef + jet_tree_nch_lorenz4_eq_node(y->nsymb,y->deg), xc = x->coef;\
       yc < yf;\
       ++yc, ++xc){myscal_axpy_si((*yc),s,a,(*xc));}\
}

/* h <- a + b */
void jet_tree_add2_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  // assert(h && a && b);
  // assert(h->nsymb == a->nsymb && p->nsymb == b->nsymb);
  // assert(h->deg == a->deg && h->deg == b->deg);
  int i, k = jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
  for (i = 0; i < k; ++i)
  {myscal_add2(h->coef[i],a->coef[i],b->coef[i]);}
}

/* h <- a - b */
static void jet_tree_sub2_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  // assert(h && a && b); // assert(h != b);
  // assert(h->nsymb == a->nsymb && h->nsymb == b->nsymb);
  // assert(h->deg == a->deg && h->deg == b->deg);
  int i, k = jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
  for (i = 0; i < k; ++i)
  {myscal_sub2(h->coef[i],a->coef[i],b->coef[i]);}
}

/* h <- h + a*b */
void jet_tree_pph_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  // assert(h && a && b); //// assert(h != a && h != b);
  if (h->nsymb > 2)
    {
      __typeof__(h->coef) hc, hf, ac;
      if (a->deg > 0)
        {
          if (b->deg > 0)
            {
              __typeof__(*h) *aa, *bb, *hh, *hh0, *af, *bf;
              af = a->term + a->deg; bf = b->term + b->deg;
              for (aa = a->term, hh0 = h->term; aa < af; ++aa, ++hh0)
                {
                  for (bb = b->term, hh = hh0;
                       bb < bf;
                       jet_tree_pph_lorenz4_eq_node(hh, aa, bb), ++bb, ++hh){}
//                  // assert(pp <= p->term + p->deg);
                  jet_tree_axpy_lorenz4_eq_node(hh,*(bb->coef),aa, hc,hf,ac);
                }
//              // assert(pp0 <= p->term + p->deg);
              for (bb = b->term, hh = hh0;
                   bb < bf;
                   ++bb, ++hh){jet_tree_axpy_lorenz4_eq_node(hh,*(aa->coef),bb, hc,hf,ac);}
              myscal_axpy(*(hh->coef),*(aa->coef),*(bb->coef));
            }
          else jet_tree_axpy_lorenz4_eq_node(h,*(b->coef),a, hc,hf,ac);
        }
      else jet_tree_axpy_lorenz4_eq_node(h,*(a->coef),b, hc,hf,ac);
    }
  else if (h->nsymb == 2)
    {
      __typeof__(*h->coef) *aa, *bb, *hh, *hh0, *af, *bf;
      af = a->coef + a->deg; bf = b->coef + b->deg;
      for (aa = a->coef, hh0 = h->coef; aa <= af; ++aa, ++hh0)
        for (bb = b->coef, hh = hh0;
             bb <= bf;
             ++bb, ++hh){myscal_axpy((*hh),(*aa),(*bb));}
    }
  else myscal_axpy(*(h->coef), *(a->coef), *(b->coef));
}

/* h <- h - a*b */
void jet_tree_mph_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  // assert(h && a && b); //// assert(h != a && h != b);
  if (h->nsymb > 2)
    {
      __typeof__(h->coef) hc, hf, ac;
      if (a->deg > 0)
        {
          if (b->deg > 0)
            {
              __typeof__(*h) *aa, *bb, *hh, *hh0, *af, *bf;
              af = a->term + a->deg; bf = b->term + b->deg;
              for (aa = a->term, hh0 = h->term; aa < af; ++aa, ++hh0)
                {
                  for (bb = b->term, hh = hh0;
                       bb < bf;
                       jet_tree_mph_lorenz4_eq_node(hh, aa, bb), ++bb, ++hh){}
//                  // assert(pp <= p->term + p->deg);
                  jet_tree_axmy_lorenz4_eq_node(hh,*(bb->coef),aa, hc,hf,ac);
                }
//              // assert(pp0 <= p->term + p->deg);
              for (bb = b->term, hh = hh0;
                   bb < bf;
                   ++bb, ++hh){jet_tree_axmy_lorenz4_eq_node(hh,*(aa->coef),bb, hc,hf,ac);}
              myscal_axmy(*(hh->coef),*(aa->coef),*(bb->coef));
            }
          else jet_tree_axmy_lorenz4_eq_node(h,*(b->coef),a, hc,hf,ac);
        }
      else jet_tree_axmy_lorenz4_eq_node(h,*(a->coef), b, hc,hf,ac);
    }
  else if (h->nsymb == 2)
    {
      __typeof__(*h->coef) *aa, *bb, *hh, *hh0, *af, *bf;
      af = a->coef + a->deg; bf = b->coef + b->deg;
      for (aa = a->coef, hh0 = h->coef; aa <= af; ++aa, ++hh0)
        for (bb = b->coef, hh = hh0;
             bb <= bf;
             ++bb, ++hh) {myscal_axmy((*hh), (*aa), (*bb));}
    }
  else myscal_axmy(*(h->coef),*(a->coef),*(b->coef));
}

/* h <- h + s*a*b */
void jet_tree_pph_scal_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, myscal_t scal, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  if (h->nsymb > 2)
    {
      __typeof__(h->coef) hc, hf, ac;
      if (a->deg > 0)
        {
          if (b->deg > 0)
            {
              __typeof__(*h) *aa, *bb, *hh, *hh0, *af, *bf;
              af = a->term + a->deg; bf = b->term + b->deg;
              for (aa = a->term, hh0 = h->term; aa < af; ++aa, ++hh0)
                {
                  for (bb = b->term, hh = hh0;
                       bb < bf;
                       jet_tree_pph_scal_lorenz4_eq_node(hh, scal, aa, bb), ++bb, ++hh){}
//                  // assert(pp <= p->term + p->deg);
                  jet_tree_axpy_scal_lorenz4_eq_node(hh,scal,*(bb->coef),aa, hc, hf, ac);
                }
//              // assert(pp0 <= p->term + p->deg);
              for (bb = b->term, hh = hh0;
                   bb < bf;
                   ++bb, ++hh){jet_tree_axpy_scal_lorenz4_eq_node(hh,scal,*(aa->coef),bb, hc, hf, ac);}
              myscal_axpy_scal(*(hh->coef),scal,*(aa->coef),*(bb->coef));
            }
          else jet_tree_axpy_scal_lorenz4_eq_node(h,scal,*(b->coef),a, hc, hf, ac);
        }
      else {jet_tree_axpy_scal_lorenz4_eq_node(h,scal,*(a->coef),b, hc, hf, ac);}
    }
  else if (h->nsymb == 2)
    {
      __typeof__(*h->coef) *aa, *bb, *hh, *hh0, *af, *bf;
      af = a->coef + a->deg; bf = b->coef + b->deg;
      for (aa = a->coef, hh0 = h->coef; aa <= af; ++aa, ++hh0)
        for (bb = b->coef, hh = hh0;
             bb <= bf;
             ++bb, ++hh){myscal_axpy_scal((*hh), scal, (*aa), (*bb));}
    }
  else {myscal_axpy_scal(*(h->coef),scal,*(a->coef),*(b->coef));}
}

void jet_tree_pph_si_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, int scal, jet_tree_ptr_lorenz4_eq_node a, jet_tree_ptr_lorenz4_eq_node b)
{
  // assert(h && a && b); //// assert(h != a && h != b);
  if (h->nsymb > 2)
    {
      __typeof__(h->coef) hc, hf, ac;
      if (a->deg > 0)
        {
          if (b->deg > 0)
            {
              __typeof__(*h) *aa, *bb, *hh, *hh0, *af, *bf;
              af = a->term + a->deg; bf = b->term + b->deg;
              for (aa = a->term, hh0 = h->term; aa < af; ++aa, ++hh0)
                {
                  for (bb = b->term, hh = hh0;
                       bb < bf;
                       jet_tree_pph_si_lorenz4_eq_node(hh, scal, aa, bb), ++bb, ++hh){}
//                  // assert(pp <= p->term + p->deg);
                  jet_tree_axpy_si_lorenz4_eq_node(hh,scal,*(bb->coef),aa, hc, hf, ac);
                }
//              // assert(pp0 <= p->term + p->deg);
              for (bb = b->term, hh = hh0;
                   bb < bf;
                   ++bb, ++hh){jet_tree_axpy_si_lorenz4_eq_node(hh,scal,*(aa->coef),bb, hc, hf, ac);}
              myscal_axpy_si(*(hh->coef),scal,*(aa->coef),*(bb->coef));
            }
          else jet_tree_axpy_si_lorenz4_eq_node(h,scal,*(b->coef),a, hc, hf, ac);
        }
      else {jet_tree_axpy_si_lorenz4_eq_node(h,scal,*(a->coef),b, hc, hf, ac);}
    }
  else if (h->nsymb == 2)
    {
      __typeof__(*h->coef) *aa, *bb, *hh, *hh0, *af, *bf;
      af = a->coef + a->deg; bf = b->coef + b->deg;
      for (aa = a->coef, hh0 = h->coef; aa <= af; ++aa, ++hh0)
        for (bb = b->coef, hh = hh0;
             bb <= bf;
             ++bb, ++hh){myscal_axpy_si((*hh), scal, (*aa), (*bb));}
    }
  else {myscal_axpy_si(*(h->coef),scal,*(a->coef),*(b->coef));}
}

#define jet_tree_mul2_scal_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_mul2((*hc),(*ac),s);}\
}
#define jet_tree_mul2_si_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_mul2_si((*hc),(*ac),s);}\
}
#define jet_tree_mul2_d_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_mul2_d((*hc),(*ac),s);}\
}

#define jet_tree_div2_scal_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_div2((*hc),(*ac),s);}\
}
#define jet_tree_div2_si_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_div2_si((*hc),(*ac),s);}\
}
#define jet_tree_div2_d_lorenz4_eq_node(h,a,s, hc,hf,ac) {\
  for (hc = (h).coef, hf = hc + jet_tree_nch_lorenz4_eq_node((h).nsymb, (h).deg), ac = (a).coef;\
       hc < hf;\
       ++hc, ++ac){myscal_div2_d((*hc),(*ac),s);}\
}

#define get_zero_coef(x) x->coef

myscal_t* jet_tree_get_coef_lorenz4_eq_node(jet_tree_ptr_lorenz4_eq_node h, const int idx[])
{
  int k;
  for (k = h->nsymb-1; k >= 0; --k)
    {
      if (h->nsymb==1 || h->deg==0) break;
      h = h->term+idx[k];
    }
  return h->coef;
}

void jet_tree_sumeval_lorenz4_eq_node(myscal_t val[1], jet_tree_ptr_lorenz4_eq_node h, myscal_t x[])
{
  int n, k, m;
  myscal_t y, x1, xk;
  myscal_init(y); myscal_set_zero(y);
  myscal_init(x1);
  myscal_init(xk);
  n= h->nsymb;
  k= h->deg;
  myscal_set(x1,x[n-1]);
  if (k== 0) {
    myscal_add2(y,y,*(h->coef));
  } else if (n==1) {
    myscal_set_si(xk,1);
    for (m = 0; m < k; ++m) {
      myscal_mul2(xk,xk,x1);
    }
    myscal_mul2(y,*(h->coef),xk);
  } else {
    jet_tree_sumeval_lorenz4_eq_node(&y,(h->term)+k, x);
    for (m= k; m>0; m--) {
      myscal_mul2(y,y,x1);
      jet_tree_sumeval_lorenz4_eq_node(&y,h->term+m-1, x);
    }
  }
  myscal_add2(*val,*val,y);
  myscal_clean(y);
  myscal_clean(x1);
  myscal_clean(xk);
}

void jet_tree_fprintf_lorenz4_eq_node(FILE *const file, const char *fmt, jet_tree_ptr_lorenz4_eq_node h)
{
  // assert(h && h->coef);
  myscal_t *hc, *hf;

  for (hc = h->coef, hf = hc + jet_tree_nch_lorenz4_eq_node(h->nsymb, h->deg);
       hc < hf;
       myscal_fprintf(file, fmt, *hc), ++hc){}
}



size_t jet_tree_init_lorenz4_eq(jet_tree_ptr_lorenz4_eq s)
{
  // assert(s);
  int i, j;
  myscal_t *coef=NULL;
  size_t mem, k;

  k = (max_deg + 1) * sizeof(**s);
  *s = (__typeof__(*s)) malloc(k);
  if (*s==NULL) {
    fprintf(stderr,"%s:%d Not enough memory\n",__FILE__,__LINE__);
    fflush(stderr); exit(2);
   }
  mem = k;
  k = _JET_COEFFICIENTS_COUNT_TOTAL_;
  coef = (__typeof__(coef)) malloc(k * sizeof(*coef));
  if (coef==NULL) {
    fprintf(stderr,"%s:%d Not enough memory\n",__FILE__,__LINE__);
    fflush(stderr); exit(2);
   }
  for (j = 0; j < k; ++j) {myscal_init(coef[j]);}
  mem+= k * sizeof(*coef);
  for (i = 0; i <= max_deg; ++i)
    {
      /*k = jet_tree_nch_lorenz4_eq_node(max_nsymb, i);
      coef = (__typeof__(coef)) malloc(k * sizeof(*coef));
      if (coef==NULL) {
        fprintf(stderr,"%s:%d Not enough memory\n",__FILE__,__LINE__);
        fflush(stderr); exit(2);
       }
      for (j = 0; j < k; ++j) {myscal_init(coef[j]);}
      mem += k * sizeof(*coef);*/
      mem += jet_tree_init_lorenz4_eq_node((*s) + i, max_nsymb, i, coef);
      coef+= jet_tree_nch_lorenz4_eq_node(max_nsymb, i);
    }
  return mem;
}

void jet_tree_initup_lorenz4_eq(int ns, int dg)
{
#pragma omp single
{
  if (ns > max_nsymb) {
    fprintf(stderr, "%s:%d nsymb value illegal %d > %d\n", __FILE__,__LINE__, ns, max_nsymb);
    fflush(stderr); exit(1);
  }
  if (dg > max_deg) {
    fprintf(stderr, "%s:%d deg value illegal %d > %d\n", __FILE__,__LINE__, dg, max_deg);
    fflush(stderr); exit(1);
  }
  /*max_nsymb = ns;
  max_deg = dg;*/
}
  nsymb = ns;
  deg = dg;
  jet_tree_init_lorenz4_eq(&my_jet_tmp);
  myscal_init(my_scal_tmp);
}

void jet_tree_clean_lorenz4_eq(jet_tree_ptr_lorenz4_eq s)
{
  if (s==NULL || *s==NULL) return;

  int j, k = _JET_COEFFICIENTS_COUNT_TOTAL_;
  __typeof__(*s) st, sf;
  for (st = *s, sf = st + max_deg;
       st < sf;
       ++st) {
    jet_tree_clean_lorenz4_eq_node(st);
    /*free(st->coef); st->coef = NULL;*/
  }
  jet_tree_clean_lorenz4_eq_node(st);
  /*free(st->coef); st->coef = NULL;*/
  for (j = 0; j < k; ++j) {myscal_clean((*s)->coef[j]);}
  free((*s)->coef); (*s)->coef = NULL;

  free(*s); *s = NULL;
}

void jet_tree_cleanup_lorenz4_eq(void)
{
  myscal_clean(my_scal_tmp);
  jet_tree_clean_lorenz4_eq(&my_jet_tmp);
  nsymb = 0;
  deg = 0;
/*#pragma omp single
{
  max_nsymb = 0;
  max_deg = 0;
}*/
}

int jet_tree_set_num_symbs_lorenz4_eq(int nsymbs)
{
  fprintf(stderr,"%s:%d Action not allowed\n",__FILE__,__LINE__); fflush(stderr); exit(1);
  return 0;
}

int jet_tree_set_deg_lorenz4_eq(int degree)
{
  int k=deg;
  if (degree > max_deg) {
    fprintf(stderr,"%s:%d Invalid degree %d > %d\n",__FILE__,__LINE__,degree,max_deg);
    fflush(stderr); exit(1);
  }
  deg = degree;
  return k;
}

void jet_tree_set_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a); // assert(s->nsymb == a->nsymb); // assert(deg == deg);
  if (s == a) return;

  __typeof__(s) st, at, sf;
  for (st = s, sf = st + deg, at= a;
       st < sf;
       jet_tree_set_lorenz4_eq_node(st,at), ++st, ++at){}
  jet_tree_set_lorenz4_eq_node(st,at);
}

void jet_tree_set_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, myscal_t scal)
{
  // assert(s && *s);
  __typeof__(s) st, sf;

  jet_tree_set_scal_lorenz4_eq_node(s,scal);
  for (st = s, sf = st + deg;
       st < sf;
       ++st, jet_tree_set_si_lorenz4_eq_node(st,0)){}
}

void jet_tree_set_d_lorenz4_eq(jet_tree_t_lorenz4_eq s, double d)
{
  // assert(s && *s);
  __typeof__(s) st, sf;

  jet_tree_set_d_lorenz4_eq_node(s,d);
  for (st = s, sf = st + deg;
       st < sf;
       ++st, jet_tree_set_si_lorenz4_eq_node(st,0)){}
}


void jet_tree_set_si_lorenz4_eq(jet_tree_t_lorenz4_eq s, int d)
{
  // assert(s && *s);
  __typeof__(s) st, sf;

  jet_tree_set_si_lorenz4_eq_node(s,d);
  for (st = s, sf = st + deg;
       st < sf;
       ++st, jet_tree_set_si_lorenz4_eq_node(st,0)){}
}
void jet_tree_set_scal_array_lorenz4_eq(jet_tree_t_lorenz4_eq s,myscal_t *b)
{
  int i;

  for (i=0;i <_JET_COEFFICIENTS_COUNT_TOTAL_;
        b++, i++){myscal_set( MY_JET_DATA(s,i) ,*b);}
}

void jet_tree_set_jet_2_scal_array_lorenz4_eq(myscal_t *b, jet_tree_t_lorenz4_eq s)
{
 int i;

  for (i=0 ; i < _JET_COEFFICIENTS_COUNT_TOTAL_;
        b++, i++){myscal_set(*b, MY_JET_DATA(s,i) );}
}

void jet_tree_to_scal_lorenz4_eq(myscal_t coef[1], jet_tree_t_lorenz4_eq s)
{  myscal_set(*coef,*get_zero_coef(s));}

void jet_tree_get_coef_lorenz4_eq(myscal_t val[1], jet_tree_t_lorenz4_eq s, const int idx[])
{
  int i, k;
  for (i = 0, k = 0; i < s->nsymb; k += idx[i], ++i){}
  
  myscal_set(*val,*jet_tree_get_coef_lorenz4_eq_node(s+k, idx));
}

void jet_tree_set_coef_lorenz4_eq(jet_tree_t_lorenz4_eq s, const int idx[], myscal_t val)
{
  int i, k;
  for (i = 0, k = 0; i < s->nsymb; k += idx[i], ++i){}
  
  myscal_set(*jet_tree_get_coef_lorenz4_eq_node(s+k, idx),val);
}

void jet_tree_nrminf_lorenz4_eq(myscal_t nrm[1], jet_tree_t_lorenz4_eq s)
{
  __typeof__(s) st, sf;
  myscal_set_si(*nrm,0);
  for (st = s, sf = st + deg;
       st < sf;
       jet_tree_nrminf_lorenz4_eq_node(nrm,st), ++st){}
   jet_tree_nrminf_lorenz4_eq_node(nrm,st);
}

void jet_tree_nrm2_lorenz4_eq(myscal_t nrm[1], jet_tree_t_lorenz4_eq s)
{
  __typeof__(s) st, sf;
  myscal_set_si(*nrm,0);
  for (st = s, sf = st + deg;
       st < sf;
       jet_tree_nrm2_lorenz4_eq_node(nrm,st), ++st){}
   jet_tree_nrm2_lorenz4_eq_node(nrm,st);
}

void jet_tree_neg_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a )
{
  // assert(s && a); // assert(s->nsymb == a->nsymb); // assert(deg == deg);
  // assert(*s && *a);
  if (s == a) return;

  __typeof__(s) st, at, sf;
  for (st = s, sf = st + deg, at = a;
       st < sf;
       jet_tree_neg_lorenz4_eq_node(st,at), ++st, ++at){}
  jet_tree_neg_lorenz4_eq_node(st,at);
}

void jet_tree_add2_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, jet_tree_t_lorenz4_eq b)
{
  // assert(s && a && b);

  int i;
  for (i = 0; i <= deg; ++i) {jet_tree_add2_lorenz4_eq_node(s+i, a+i, b+i);}
}

void jet_tree_sub2_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, jet_tree_t_lorenz4_eq b)
{
  // assert(s && a && b);

  int i;
  for (i = 0; i <= deg; ++i) jet_tree_sub2_lorenz4_eq_node(s+i, a+i, b+i);
}

void jet_tree_set_sq_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a); // assert(s != a);
  // assert(*s && *a);

  myscal_t *a0, *sc, *sf;
  sc = get_zero_coef(s);
  a0 = get_zero_coef(a);
  myscal_mul2((*sc),(*a0),(*a0));

  int i, j, k, ord=deg;
  if (ord % 2)
    {
      i = ord;
      /* odd case */
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      k = i/2;
      for (j = 0; j <= k; ++j) jet_tree_pph_lorenz4_eq_node(s+i, a+j, a+(i-j));
      jet_tree_mul2_si_lorenz4_eq_node(s[i], s[i], 2, sc,sf,a0);
      --ord;
    }
  for (i = 1; i <= ord; ++i)
    {
       /* odd case */
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      k = i/2;
      for (j = 0; j <= k; ++j) jet_tree_pph_lorenz4_eq_node(s+i, a+j, a+(i-j));
      jet_tree_mul2_si_lorenz4_eq_node(s[i], s[i], 2, sc,sf,a0);

      ++i;

      /* even case */
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      k = i/2;
      for (j = 0; j < k; ++j) jet_tree_pph_lorenz4_eq_node(s+i, a+j, a+(i-j));
      jet_tree_mul2_si_lorenz4_eq_node(s[i], s[i], 2, sc,sf,a0);
      jet_tree_pph_lorenz4_eq_node(s+i, a + k, a + k);
    }
}

void jet_tree_mul2_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, jet_tree_t_lorenz4_eq b)
{
// assert(s && a && b); // assert(s != a && s != b);
// assert(*s && *a && *b);
  int i, j;
  myscal_t *sc,*sf;

  if (a == b) {jet_tree_set_sq_lorenz4_eq(s,a); return;}
  for (i = 0; i <= deg; ++i)
    {
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      for (j = 0; j <= i; ++j)
        {jet_tree_pph_lorenz4_eq_node(s+i, a+j, b+(i-j));}
    }
}

void jet_tree_div2_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, jet_tree_t_lorenz4_eq b)
{
  // assert(s && a && b); // assert(s != b);
  // assert(*s && *a && *b);
  myscal_t *b0, *sc,*sf,*sc1;
  int i, j;
  sc = get_zero_coef(a);
  b0 = get_zero_coef(b);
  sf = get_zero_coef(s);
  myscal_div2(*sf,*sc,*b0);
  for (i = 1; i <= deg; ++i)
    {
      jet_tree_set_lorenz4_eq_node(s+i, a+i);
      for (j = 0; j < i; ++j) {jet_tree_mph_lorenz4_eq_node(s+i, s+j, b + (i-j));}
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i],(*b0), sc,sf,sc1);
    }
}

void jet_tree_add2_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, myscal_t b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_add2(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_sub2_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, myscal_t b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_sub2(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; 
       ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_scal_sub2_lorenz4_eq(jet_tree_t_lorenz4_eq s, myscal_t b, jet_tree_t_lorenz4_eq a)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_sub2(*(s->coef),b,*(a->coef));
  for (st = s, at= a;
       st < sf; 
       ++st, ++at, jet_tree_neg_lorenz4_eq_node(st,at)){}
}

void jet_tree_mul2_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, myscal_t b)
{
  int k;
  myscal_t *sc, *scf, *ac;
  for (k = 0; k <= deg; ++k)
  {jet_tree_mul2_scal_lorenz4_eq_node(s[k],a[k],b, sc,scf,ac);}
}

void jet_tree_div2_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, myscal_t b)
{
  // assert(s && a);
  // assert(scal);

  myscal_t *sc, *scf, *ac;
  __typeof__(s) st, sf, at;
  for (st = s, sf = st + deg, at= a;
       st < sf;
       ++st, ++at) {jet_tree_div2_scal_lorenz4_eq_node(st[0], at[0], b, sc,scf,ac);}
  jet_tree_div2_scal_lorenz4_eq_node(st[0], at[0], b, sc,scf,ac);
}

void jet_tree_scal_div2_lorenz4_eq(jet_tree_t_lorenz4_eq s, myscal_t b, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a); // assert(s != a);
  // assert(scal);
  myscal_t *a0, *sc,*sf,*sc1;
  int i, j;
  a0 = get_zero_coef(a);
  sf = get_zero_coef(s);
  myscal_div2(*sf,b,*a0);
  for (i = 1; i <= deg; ++i)
    {
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      for (j = 0; j < i; ++j) {jet_tree_mph_lorenz4_eq_node(s+i, s+j, a+(i-j));}
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i],(*a0), sc,sf,sc1);
    }
}


void jet_tree_add2_d_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, double b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_add2_d(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_sub2_d_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, double b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_sub2_d(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; 
       ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_d_sub2_lorenz4_eq(jet_tree_t_lorenz4_eq s, double b, jet_tree_t_lorenz4_eq a)
{
  fprintf(stderr,"%s:%d to implement\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_mul2_d_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, double b)
{
  int k;
  myscal_t *sc, *scf, *ac;
  for (k = 0; k <= deg; ++k)
  {jet_tree_mul2_d_lorenz4_eq_node(s[k],a[k],b, sc,scf,ac);}
}

void jet_tree_div2_d_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, double b)
{
  fprintf(stderr,"%s:%d to implement\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_d_div2_lorenz4_eq(jet_tree_t_lorenz4_eq s, double b, jet_tree_t_lorenz4_eq a)
{
  fprintf(stderr,"%s:%d to implement\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_add2_si_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, int b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_add2_si(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_sub2_si_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, int b)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_sub2_si(*(s->coef),*(a->coef),b);
  if (s != a) for (st = s, at = a;
       st < sf; 
       ++st, ++at, jet_tree_set_lorenz4_eq_node(st,at)){}
}

void jet_tree_si_sub2_lorenz4_eq(jet_tree_t_lorenz4_eq s, int b, jet_tree_t_lorenz4_eq a)
{
  __typeof__(s) st, at, sf = s + deg;
  myscal_si_sub2(*(s->coef),b,*(a->coef));
  for (st = s, at= a;
       st < sf; 
       ++st, ++at, jet_tree_neg_lorenz4_eq_node(st,at)){}
//  fprintf(stderr,"%s:%d to test\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_mul2_si_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, int b)
{
  int k;
  myscal_t *sc, *scf, *ac;
  for (k = 0; k <= deg; ++k)
  {jet_tree_mul2_si_lorenz4_eq_node(s[k],a[k],b, sc,scf,ac);}
}

void jet_tree_div2_si_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, int b)
{
  myscal_t *sc, *scf, *ac;
  __typeof__(s) st, sf, at;
  for (st = s, sf = st + deg, at= a;
       st < sf;
       ++st, ++at) {jet_tree_div2_si_lorenz4_eq_node(st[0], at[0], b, sc,scf,ac);}
  jet_tree_div2_si_lorenz4_eq_node(st[0], at[0], b, sc,scf,ac);
//  fprintf(stderr,"%s:%d to test\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_si_div2_lorenz4_eq(jet_tree_t_lorenz4_eq s, int b, jet_tree_t_lorenz4_eq a)
{
  fprintf(stderr,"%s:%d to implement\n",__FILE__,__LINE__); exit(1);
}

void jet_tree_set_sqrt_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a);
  // assert(*s && *a);

  myscal_t *s0,*sc,*scf,*sc2;
  s0 = get_zero_coef(s);
  myscal_set_sqrt((*s0), *get_zero_coef(a)); // assert(fabs(*s0) > ZERO_TOL);

  int i, j, k;
  for (i = 1; i <= deg; ++i)
    {
       /* odd case */
      jet_tree_set_lorenz4_eq_node(s+i, a+i);
      jet_tree_div2_si_lorenz4_eq_node(s[i], s[i], 2, sc,scf,sc2);
      k = i/2;
      for (j = 1; j <= k; ++j) jet_tree_mph_lorenz4_eq_node(s+i, s+j, s+(i-j));
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i], *s0, sc,scf,sc2);

      if (++i > deg) break;

      /* even case */
      jet_tree_set_lorenz4_eq_node(s+i, a+i);
      k = i/2;
      jet_tree_mph_lorenz4_eq_node(s+i, s+k, s+k);
      jet_tree_div2_si_lorenz4_eq_node(s[i], s[i], 2, sc,scf,sc2);
      for (j = 1; j < k; ++j) jet_tree_mph_lorenz4_eq_node(s+i, s+j, s+(i-j));
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i], *s0, sc,scf,sc2);
    }
}

void jet_tree_set_pow_scal_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a, myscal_t scal)
{
  // assert(s && a); // assert(s != a);
//  if (scal == 2)     {jet_tree_set_sq_lorenz4_eq(s,a); return;}
//  if (scal == 0.5e0) {jet_tree_set_sqrt_lorenz4_eq(s,a); return;}
  // assert(*s && *a);

  myscal_t *a0, *s0, *tmp, *sc,*sf,*sc1;
  tmp = get_zero_coef(my_jet_tmp);
  sc = get_zero_coef(s);
  a0 = get_zero_coef(a); // assert(fabs(*a0) > ZERO_TOL);
  myscal_set_pow((*sc), *a0, scal);

  int i, j;
  for (i = 1; i <= deg; ++i)
    {
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);
      for (j = 0; j < i; ++j) {
        myscal_mul2_si(*tmp,scal,i-j);
        myscal_sub2_si(*tmp,*tmp,j);
        jet_tree_pph_scal_lorenz4_eq_node(s+i, *tmp, a+(i-j), s+j);
      }
      myscal_mul2_si(*tmp,*a0,i);
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i], (*tmp), sc,sf,sc1);
    }
}

void jet_tree_set_exp_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a); // assert(s != a);
  // assert(*s && *a);

  myscal_t *s0,*scf,*sc;
  s0 = get_zero_coef(s);
  s0 = get_zero_coef(s);
  myscal_set_exp((*s0), *get_zero_coef(a));

  int i, j;
  for (i = 1; i <= deg; ++i)
    {
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,scf);
      for (j = 0; j < i; ++j)
      {jet_tree_pph_si_lorenz4_eq_node(s+i, i-j, a+(i-j), s+j);}
      jet_tree_div2_si_lorenz4_eq_node(s[i], s[i], i, sc,scf,s0);
    }
}

void jet_tree_set_log_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  // assert(s && a); // assert(s != a);
  // assert(*s && *a);

  myscal_t *a0, *sc,*sf,*sc1, *tmp;
  tmp = get_zero_coef(my_jet_tmp);
  sc = get_zero_coef(s);
  a0 = get_zero_coef(a);
  myscal_set_log((*sc), *get_zero_coef(a));

  int i, j;
  for (i = 1; i <= deg; ++i)
    {
      jet_tree_set_lorenz4_eq_node(s+i, a+i);
      for (j = 1; j < i; ++j) {
        myscal_set_si(*tmp, -j);
        myscal_div2_si(*tmp, *tmp, i);
        jet_tree_pph_scal_lorenz4_eq_node(s+i, *tmp, a+(i-j), s+j);
      }
      jet_tree_div2_scal_lorenz4_eq_node(s[i], s[i], *a0, sc,sf,sc1);
    }
}

#define jet_tree_sincos_lorenz4_eq(s,c,a)\
{\
  /* assert(s && c && a); assert(s != a && s != c && c != a);*/\
  /* assert(*s && *c && *a);*/\
\
  int i, j;\
  myscal_t aux1,aux2;\
  myscal_t *a0, *sc, *sf;\
  myscal_init(aux1);\
  myscal_init(aux2);\
  a0 = get_zero_coef(a);\
  myscal_set_sin((*get_zero_coef(s)),(*a0));\
  myscal_set_cos((*get_zero_coef(c)),(*a0));\
\
  for (i = 1; i <= deg; ++i)\
    {\
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);\
      jet_tree_set_zero_lorenz4_eq_node(c[i], sc,sf);\
\
      for (j = 0; j < i; ++j)\
        {\
          myscal_set_si(aux1,j);\
          myscal_div2_si(aux1,aux1,i);\
          myscal_sub2_si(aux2,aux1,1);\
          myscal_si_sub2(aux1,1,aux1);\
\
          jet_tree_pph_scal_lorenz4_eq_node(s+i, aux1, a+(i-j), c+j);\
          jet_tree_pph_scal_lorenz4_eq_node(c+i, aux2, a+(i-j), s+j);\
        }\
    }\
  myscal_clean(aux2);\
  myscal_clean(aux1);\
}\

void jet_tree_set_sin_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  jet_tree_sincos_lorenz4_eq(s,my_jet_tmp,a);
}

void jet_tree_set_cos_lorenz4_eq(jet_tree_t_lorenz4_eq c, jet_tree_t_lorenz4_eq a)
{
  jet_tree_sincos_lorenz4_eq(my_jet_tmp,c,a);
}

void jet_tree_set_tan_lorenz4_eq(jet_tree_t_lorenz4_eq t, jet_tree_t_lorenz4_eq a)
{
  int j,k,ii,jj ,nk;
  myscal_t *t0,*tcf,*tc;
  myscal_set_tan(*get_zero_coef(t),*get_zero_coef(a));
  myscal_mul2(*get_zero_coef(my_jet_tmp),*get_zero_coef(t),*get_zero_coef(t));

  nk = (deg+1) - ((deg+1)%2==0);
  for (k = 1; k < nk; k++) {
    jet_tree_set_zero_lorenz4_eq_node(t[k], tc,tcf);
    for (j = 0; j < k; j++) 
    {jet_tree_pph_si_lorenz4_eq_node(t+k,k-j,a+(k-j),my_jet_tmp+j);}
    jet_tree_div2_si_lorenz4_eq_node(t[k],t[k],k, tc,tcf,t0);
    jet_tree_add2_lorenz4_eq_node(t+k,t+k,a+k);

    /* k odd */
    jet_tree_set_zero_lorenz4_eq_node(my_jet_tmp[k], tc,tcf);
    for (j = 0; j <= (k-1)/2; j++)
    {jet_tree_pph_lorenz4_eq_node(my_jet_tmp+k,t+j,t+(k-j));}
    jet_tree_mul2_si_lorenz4_eq_node(my_jet_tmp[k],my_jet_tmp[k],2, tc,tcf,t0);

    ++k;

    jet_tree_set_zero_lorenz4_eq_node(t[k], tc,tcf);
    for (j = 0; j < k; j++) 
    {jet_tree_pph_si_lorenz4_eq_node(t+k,k-j,a+(k-j),my_jet_tmp+j);}
    jet_tree_div2_si_lorenz4_eq_node(t[k],t[k],k, tc,tcf,t0);
    jet_tree_add2_lorenz4_eq_node(t+k,t+k,a+k);

    /* k even */
    jet_tree_set_zero_lorenz4_eq_node(my_jet_tmp[k], tc,tcf);
    for (j = 0; j <= (k-2)/2; j++)
    {jet_tree_pph_lorenz4_eq_node(my_jet_tmp+k,t+j,t+(k-j));}
    jet_tree_mul2_si_lorenz4_eq_node(my_jet_tmp[k],my_jet_tmp[k],2, tc,tcf,t0);

    j = k/2;
    jet_tree_pph_lorenz4_eq_node(my_jet_tmp+k,t+j,t+j);
  }
  if ((deg+1)%2==0) {
    k = deg;
    jet_tree_set_zero_lorenz4_eq_node(t[k], tc,tcf);
    for (j = 0; j < k; j++) 
    {jet_tree_pph_si_lorenz4_eq_node(t+k,k-j,a+(k-j),my_jet_tmp+j);}
    jet_tree_div2_si_lorenz4_eq_node(t[k],t[k],k, tc,tcf,t0);
    jet_tree_add2_lorenz4_eq_node(t+k,t+k,a+k);
  }

//  fprintf(stderr,"%s:%d to test\n",__FILE__,__LINE__); exit(1);
}
void jet_tree_set_atan_lorenz4_eq(jet_tree_t_lorenz4_eq t, jet_tree_t_lorenz4_eq a)
{
  int j,k,ii,jj;
  myscal_t *t0,*tcf,*tc;
  myscal_set_atan(*get_zero_coef(t),*get_zero_coef(a));
  jet_tree_set_sq_lorenz4_eq(my_jet_tmp,a);
  jet_tree_add2_si_lorenz4_eq(my_jet_tmp,my_jet_tmp,1);
  for (k = 1; k <= deg; k++) {
    jet_tree_set_zero_lorenz4_eq_node(t[k], tc,tcf);
    for (j = 1; j < k; j++) 
    {jet_tree_pph_si_lorenz4_eq_node(t+k, j, my_jet_tmp+(k-j), t+j);}
    jet_tree_div2_si_lorenz4_eq_node(t[k],t[k],k, tc,tcf,t0);
    jet_tree_sub2_lorenz4_eq_node(t+k,a+k,t+k);
    jet_tree_div2_scal_lorenz4_eq_node(t[k],t[k],*get_zero_coef(my_jet_tmp), tc,tcf,t0);
  }
//  fprintf(stderr,"%s:%d to test\n",__FILE__,__LINE__); exit(1);
}

#define jet_tree_sinhcosh_lorenz4_eq(s,c,a)\
{\
  /* assert(s && c && a); assert(s != a && s != c && c != a);*/\
  /* assert(*s && *c && *a);*/\
\
  int i, j;\
  myscal_t aux1;\
  myscal_t *a0, *sc,*sf;\
  myscal_init(aux1);\
  a0 = get_zero_coef(a);\
  myscal_set_sinh((*get_zero_coef(s)),(*a0));\
  myscal_set_cosh((*get_zero_coef(c)),(*a0));\
\
  for (i = 1; i <= deg; ++i)\
    {\
      jet_tree_set_zero_lorenz4_eq_node(s[i], sc,sf);\
      jet_tree_set_zero_lorenz4_eq_node(c[i], sc,sf);\
\
      for (j = 0; j < i; ++j)\
        {\
          myscal_set_si(aux1,j);\
          myscal_div2_si(aux1,aux1,i);\
          myscal_si_sub2(aux1,1,aux1);\
\
          jet_tree_pph_scal_lorenz4_eq_node(s+i, aux1, a+(i-j), c+j);\
          jet_tree_pph_scal_lorenz4_eq_node(c+i, aux1, a+(i-j), s+j);\
        }\
    }\
  myscal_clean(aux1);\
}\

void jet_tree_set_sinh_lorenz4_eq(jet_tree_t_lorenz4_eq s, jet_tree_t_lorenz4_eq a)
{
  jet_tree_sinhcosh_lorenz4_eq(s,my_jet_tmp,a);
}

void jet_tree_set_cosh_lorenz4_eq(jet_tree_t_lorenz4_eq c, jet_tree_t_lorenz4_eq a)
{
  jet_tree_sinhcosh_lorenz4_eq(my_jet_tmp,c,a);
}

void jet_tree_eval_lorenz4_eq(myscal_t val[1], jet_tree_t_lorenz4_eq x, myscal_t s[])
{
  int k, m;
  myscal_set_zero(*val);
  for (k = deg; k > 0; --k) {
    jet_tree_sumeval_lorenz4_eq_node (val, x + k, s);
  }
  myscal_add2(*val,*val, *(get_zero_coef(x)));
}

void jet_tree_fprintf_lorenz4_eq(FILE *file, const char *fmt, jet_tree_t_lorenz4_eq s)
{
  int i;
  for (i=0;i <_JET_COEFFICIENTS_COUNT_TOTAL_; i++){myscal_fprintf(file, fmt,  MY_JET_DATA(s,i) );
 }
}
#undef my_jet_tmp
#undef my_scal_tmp


/* END CODE jet_tree_t_lorenz4_eq */

/***********************************************************************
 *
 * Procedure generated by the TAYLOR translator. Do not edit!
 *
 * It needs the header file 'taylor.h' to compile.
 * Run taylor with the -header -o taylor.h option to generate a sample 'taylor.h'

 * Translation info is at the end of this file.
 * Version 2.0.0 , Sept 8, 2022
 ***********************************************************************/

#include <stdio.h>
#include <stdlib.h>
MY_FLOAT **taylor_coefficients_lorenz4_eq_A(MY_FLOAT t, MY_FLOAT *x, int order, int rflag, MY_JET *sIn, MY_JET ***sOut)
{
   /* input: 
      t:     current value of the time variable 
      x:     array represent values of the state variables
      order: order of the taylor coefficients sought
      rflag: recompute flag. If you call this routine with one order 
             first, but then decided that you need a higher order of the
             taylor polynomial, you can pass 0 to rflag. This routine 
             will try to use the values already computed. Provided that 
             both x and t have not been changed, and you did not modify 
             the jet derivatives from the previous call.
      sIn:   array representing the input jet variables
             If NULL, the x values are assigned.
      sOut:  Pointer output jet variables
             If NULL, the pointer is not modified.
      Return Value:
            Two D Array, rows are the taylor coefficients of the
            state variables

     */
    extern MY_FLOAT RR;
    extern MY_FLOAT SS;
#pragma omp threadprivate(RR)

    static MY_FLOAT     _jz_cvars[2];
    #pragma omp threadprivate( _jz_cvars)
    static MY_FLOAT     *_jz_jet[13],  *_jz_save = NULL, *_jz_oneOverN=NULL,*_jz_theNs=NULL;
    static MY_FLOAT     _jz_tvar1, _jz_tvar2, _jz_tvar3, _jz_tvar4; /* tmp vars */
    static MY_FLOAT     _jz_uvar1, _jz_uvar2; /* tmp vars */
    static MY_FLOAT     _jz_svar1, _jz_svar2, _jz_svar3, _jz_svar4, _jz_svar5; /* tmp vars */
    static MY_FLOAT     _jz_wvar3, _jz_wvar4; /* tmp vars */
    static MY_FLOAT     _jz_zvar1, _jz_zvar2; /* tmp vars */
    static MY_FLOAT     _jz_MyFloatZERO, _kthtmp;
    #pragma omp threadprivate(_jz_jet, _jz_save, _jz_oneOverN, _jz_theNs, _jz_tvar1, _jz_tvar2, _jz_tvar3, _jz_tvar4, _jz_uvar1, _jz_uvar2, _jz_svar1, _jz_svar2, _jz_svar3, _jz_svar4, _jz_svar5, _jz_wvar3, _jz_wvar4, _jz_zvar1, _jz_zvar2, _jz_MyFloatZERO, _kthtmp)
/* series bgn */
    static MY_JET       *_sjz_jet[13], *_sjz_save = NULL;
    static MY_JET     _sjz_tvar1, _sjz_tvar2, _sjz_tvar3, _sjz_tvar4; /* tmp vars */
    static MY_JET     _sjz_uvar1, _sjz_uvar2; /* tmp vars */
    static MY_JET     _sjz_svar1, _sjz_svar2, _sjz_svar3, _sjz_svar4, _sjz_svar5; /* tmp vars */
    static MY_JET     _sjz_wvar3, _sjz_wvar4; /* tmp vars */
    static MY_JET     _sjz_zvar1, _sjz_zvar2; /* tmp vars */
    static MY_JET     _kthStmp; /* tmp vars */
    #pragma omp threadprivate(_sjz_jet, _sjz_save, _sjz_tvar1, _sjz_tvar2, _sjz_tvar3, _sjz_tvar4, _sjz_uvar1, _sjz_uvar2, _sjz_svar1, _sjz_svar2, _sjz_svar3, _sjz_svar4, _sjz_svar5,  _sjz_wvar3, _sjz_wvar4, _sjz_zvar1, _sjz_zvar2, _kthStmp)
/* series end */
    static int          _jz_maxOrderUsed  = -1;
    static int          _jz_lastOrder = 0, _jz_initialized=0, _jz_ginitialized=0;
    int                 _jz_i, _jz_j, _jz_k, _jz_l, _jz_m, _jz_n, _jz_oorder ;
    #pragma omp threadprivate(_jz_maxOrderUsed,_jz_lastOrder, _jz_initialized, _jz_ginitialized)
    /* allocating memory if needed */
    if (_jz_maxOrderUsed < order )  { 
    	 if (_jz_ginitialized == 0) { 
    	   InitMyFloat(_jz_tvar1); InitMyFloat(_jz_tvar2);InitMyFloat(_jz_tvar3);InitMyFloat(_jz_tvar4);
    	   InitMyFloat(_jz_svar1); InitMyFloat(_jz_svar2);InitMyFloat(_jz_svar3);InitMyFloat(_jz_svar4);
    	   InitMyFloat(_jz_svar5); InitMyFloat(_jz_zvar1);InitMyFloat(_jz_zvar2);
    	   InitMyFloat(_jz_uvar1); InitMyFloat(_jz_uvar2);
    	   InitMyFloat(_jz_wvar3);InitMyFloat(_jz_wvar4);
    	   InitMyFloat(_jz_MyFloatZERO); InitMyFloat(_kthtmp);
    	   MakeMyFloatC(_jz_MyFloatZERO, "0", (double)0);
    	   for(_jz_i=0; _jz_i<2; _jz_i++) {
    	       InitMyFloat(_jz_cvars[_jz_i]);
    	   }
    	   /* series bgn */
    	   InitJet(_sjz_tvar1); InitJet(_sjz_tvar2);InitJet(_sjz_tvar3);InitJet(_sjz_tvar4);
    	   InitJet(_sjz_svar1); InitJet(_sjz_svar2);InitJet(_sjz_svar3);InitJet(_sjz_svar4);
    	   InitJet(_sjz_svar5); InitJet(_sjz_zvar1);InitJet(_sjz_zvar2);
    	   InitJet(_sjz_uvar1); InitJet(_sjz_uvar2);
    	   InitJet(_sjz_wvar3); InitJet(_sjz_wvar4);
    	   InitJet(_kthStmp);
    	   /* series end */
    	 }
    	 if (rflag > 0) rflag = 0; /* have to recompute everything */
    	 _jz_oorder=_jz_maxOrderUsed;
    	 _jz_maxOrderUsed  = order;
    	 if (_jz_ginitialized) { 
    	   for(_jz_i=0; _jz_i< _jz_oorder+1; _jz_i++) {ClearMyFloat(_jz_oneOverN[_jz_i]); ClearMyFloat(_jz_theNs[_jz_i]);}
    	   free(_jz_oneOverN); free(_jz_theNs);
    	 }
    	 _jz_theNs = (MY_FLOAT *)malloc((order+1) * sizeof(MY_FLOAT));
    	 _jz_oneOverN = (MY_FLOAT *)malloc((order+1) * sizeof(MY_FLOAT));
    	 for(_jz_i=0; _jz_i<order+1; _jz_i++) {InitMyFloat(_jz_oneOverN[_jz_i]);InitMyFloat(_jz_theNs[_jz_i]);}
    	 MakeMyFloatC(_jz_theNs[0],"0.0", (double)0.0);
    	 MakeMyFloatC(_jz_uvar1,"1.0", (double)1.0);
    	 for(_jz_i = 1; _jz_i <= order; _jz_i++) {
    		 AssignMyFloat(_jz_tvar2, _jz_theNs[_jz_i-1]);
    		 AddMyFloatA(_jz_theNs[_jz_i], _jz_tvar2, _jz_uvar1);
	}
    	 AssignMyFloat(_jz_oneOverN[0],_jz_uvar1);
    	 AssignMyFloat(_jz_oneOverN[1],_jz_uvar1);
    	 for(_jz_i = 2; _jz_i <= order; _jz_i++) {
    		 DivideMyFloatA(_jz_oneOverN[_jz_i], _jz_uvar1,_jz_theNs[_jz_i]);
	}
    	 if (_jz_ginitialized) {
    	    for(_jz_i=0; _jz_i<(_jz_oorder+1)*(13); _jz_i++) { ClearMyFloat(_jz_save[_jz_i]);} free(_jz_save);
	  /* series bgn */
    	    for(_jz_i=0; _jz_i<(_jz_oorder+1)*(13); _jz_i++) { ClearJet(_sjz_save[_jz_i]);} free(_sjz_save);
	  /* series end */
    	 }
    	 _jz_save = (MY_FLOAT *)malloc((order+1)* 13 *sizeof(MY_FLOAT));
    	 for(_jz_i=0; _jz_i<(order+1)*(13); _jz_i++) { InitMyFloat(_jz_save[_jz_i]);}
    	 for(_jz_j = 0, _jz_k = 0; _jz_j < 13 ;  _jz_j++, _jz_k += order+1) { _jz_jet[_jz_j] =& (_jz_save[_jz_k]); }
    	 /*s eries bgn */
    	 _sjz_save = (MY_JET *)malloc((order+1)* 13 *sizeof(MY_JET));
    	 for(_jz_i=0; _jz_i<(order+1)*(13); _jz_i++) { InitJet(_sjz_save[_jz_i]);}
    	 for(_jz_j = 0, _jz_k = 0; _jz_j < 13 ;  _jz_j++, _jz_k += order+1) { _sjz_jet[_jz_j] =& (_sjz_save[_jz_k]); }
    	 /* series end */

    	 /* True constants, initialized only once. */
	 /* const: c_011=8.0 */
	 MakeMyFloatC(_jz_cvars[0],"8.0",(double)8.0);
	 /* const: c_012=3.0 */
	 MakeMyFloatC(_jz_cvars[1],"3.0",(double)3.0);
    }

    if (rflag) {
    	 if (rflag < 0 ) return(NULL);
    	 for(_jz_i = 0; rflag != 0 && _jz_i < 3; _jz_i++) {
    		 if (MyFloatA_NEQ_B(_jz_jet[_jz_i][0], x[_jz_i])) rflag = 0;
    	 }
    }

    if (rflag == 0) {
	 /* initialize all constant vars and state variables */
	 _jz_lastOrder = 1;
		 /* init state variable: */
	 AssignFloatToJet(_sjz_jet[0][0], x[0]);
	 AssignMyFloat(_jz_jet[0][0], x[0]);
		 /* init state variable: */
	 AssignFloatToJet(_sjz_jet[1][0], x[1]);
	 AssignMyFloat(_jz_jet[1][0], x[1]);
		 /* init state variable: */
	 AssignFloatToJet(_sjz_jet[2][0], x[2]);
	 AssignMyFloat(_jz_jet[2][0], x[2]);
/* series bgn */
	 if (sIn){
		 AssignJetToJet(_sjz_jet[0][0], sIn[0]);
		 AssignJetToFloat(_jz_jet[0][0], sIn[0]);
		 AssignJetToJet(_sjz_jet[1][0], sIn[1]);
		 AssignJetToFloat(_jz_jet[1][0], sIn[1]);
		 AssignJetToJet(_sjz_jet[2][0], sIn[2]);
		 AssignJetToFloat(_jz_jet[2][0], sIn[2]);
	 }
/* series end */
	 /* minus: v_015=(v_007-v_006) */
	 SubtractJetJetA(_sjz_jet[3][0], _sjz_jet[1][0], _sjz_jet[0][0]);
	 /* mult: v_016=(SS*v_015) */
	 MultiplyFloatJetA(_sjz_jet[4][0], SS, _sjz_jet[3][0]);
	 /* mult: v_017=(RR*v_006) */
	 MultiplyFloatJetA(_sjz_jet[5][0], RR, _sjz_jet[0][0]);
	 /* mult: v_018=(v_006*v_008) */
	 MultiplyJetJetA(_sjz_jet[6][0], _sjz_jet[0][0], _sjz_jet[2][0]);
	 /* minus: v_019=(v_017-v_018) */
	 SubtractJetJetA(_sjz_jet[7][0], _sjz_jet[5][0], _sjz_jet[6][0]);
	 /* minus: v_020=(v_019-v_007) */
	 SubtractJetJetA(_sjz_jet[8][0], _sjz_jet[7][0], _sjz_jet[1][0]);
	 /* mult: v_021=(v_006*v_007) */
	 MultiplyJetJetA(_sjz_jet[9][0], _sjz_jet[0][0], _sjz_jet[1][0]);
	 /* mult: v_022=(c_011*v_008) */
	 MultiplyFloatJetA(_sjz_jet[10][0], _jz_cvars[0], _sjz_jet[2][0]);
	 /* div: v_023=(v_022/c_012) */
	 DivideJetFloatA(_sjz_jet[11][0], _sjz_jet[10][0], _jz_cvars[1]);
	 /* minus: v_024=(v_021-v_023) */
	 SubtractJetJetA(_sjz_jet[12][0], _sjz_jet[9][0], _sjz_jet[11][0]);

	 /* the first derivative of state variables */
	 /* state variable 0: */
	 AssignJetToJet(_sjz_jet[0][1], _sjz_jet[4][0]);
	 AssignJetToFloat(_jz_jet[0][1], _sjz_jet[0][1]);
	 /* state variable 1: */
	 AssignJetToJet(_sjz_jet[1][1], _sjz_jet[8][0]);
	 AssignJetToFloat(_jz_jet[1][1], _sjz_jet[1][1]);
	 /* state variable 2: */
	 AssignJetToJet(_sjz_jet[2][1], _sjz_jet[12][0]);
	 AssignJetToFloat(_jz_jet[2][1], _sjz_jet[2][1]);
	}

	 /* compute the kth order derivatives of all vars */
	 for(_jz_k = _jz_lastOrder; _jz_k < order; _jz_k++) {
		 /* derivative for tmp variables */
		 /* minus: v_015=(v_007-v_006) */
		 SubtractJetJetA(_sjz_jet[3][_jz_k], _sjz_jet[1][_jz_k],_sjz_jet[0][_jz_k]);
		 /* mult: v_016=(SS*v_015) */
		 MultiplyFloatJetA(_sjz_jet[4][_jz_k], SS, _sjz_jet[3][_jz_k]);
		 /* mult: v_017=(RR*v_006) */
		 MultiplyFloatJetA(_sjz_jet[5][_jz_k], RR, _sjz_jet[0][_jz_k]);
		 /* mult: v_018=(v_006*v_008) */
		 { /* multiplication */
			 static MY_FLOAT tmp1, tmp2, tmp;
			 #pragma omp threadprivate(tmp1, tmp2, tmp)
			 static MY_JET stmp1, stmp2, stmp;
			 #pragma omp threadprivate(stmp1, stmp2, stmp)
			 if(_jz_initialized==0) {
			   InitMyFloat(tmp1);InitMyFloat(tmp2); InitMyFloat(tmp);
			   InitJet(stmp1);InitJet(stmp2); InitJet(stmp);
			   }
			 AssignFloatToJet(stmp, _jz_MyFloatZERO);
			 for(_jz_l=0; _jz_l<=_jz_k; _jz_l++) {
			     MultiplyJetJetA(stmp1, _sjz_jet[0][_jz_l], _sjz_jet[2][_jz_k-_jz_l]);
			     AssignJetToJet(stmp2, stmp);
			     AddJetJetA(stmp, stmp2, stmp1);
			 }
			 AssignJetToJet(_sjz_jet[6][_jz_k], stmp);
		 }
		 /* minus: v_019=(v_017-v_018) */
		 SubtractJetJetA(_sjz_jet[7][_jz_k], _sjz_jet[5][_jz_k],_sjz_jet[6][_jz_k]);
		 /* minus: v_020=(v_019-v_007) */
		 SubtractJetJetA(_sjz_jet[8][_jz_k], _sjz_jet[7][_jz_k],_sjz_jet[1][_jz_k]);
		 /* mult: v_021=(v_006*v_007) */
		 { /* multiplication */
			 static MY_FLOAT tmp1, tmp2, tmp;
			 #pragma omp threadprivate(tmp1, tmp2, tmp)
			 static MY_JET stmp1, stmp2, stmp;
			 #pragma omp threadprivate(stmp1, stmp2, stmp)
			 if(_jz_initialized==0) {
			   InitMyFloat(tmp1);InitMyFloat(tmp2); InitMyFloat(tmp);
			   InitJet(stmp1);InitJet(stmp2); InitJet(stmp);
			   }
			 AssignFloatToJet(stmp, _jz_MyFloatZERO);
			 for(_jz_l=0; _jz_l<=_jz_k; _jz_l++) {
			     MultiplyJetJetA(stmp1, _sjz_jet[0][_jz_l], _sjz_jet[1][_jz_k-_jz_l]);
			     AssignJetToJet(stmp2, stmp);
			     AddJetJetA(stmp, stmp2, stmp1);
			 }
			 AssignJetToJet(_sjz_jet[9][_jz_k], stmp);
		 }
		 /* mult: v_022=(c_011*v_008) */
		 MultiplyFloatJetA(_sjz_jet[10][_jz_k], _jz_cvars[0], _sjz_jet[2][_jz_k]);
		 /* div: v_023=(v_022/c_012) */
		 DivideJetFloatA(_sjz_jet[11][_jz_k], _sjz_jet[10][_jz_k], _jz_cvars[1]);
		 /* minus: v_024=(v_021-v_023) */
		 SubtractJetJetA(_sjz_jet[12][_jz_k], _sjz_jet[9][_jz_k],_sjz_jet[11][_jz_k]);
		 /* derivative of state variables */
		 _jz_m = _jz_k+1;
		 /* state variable 0: */
		 DivideJetFloatA(_sjz_jet[0][_jz_m], _sjz_jet[4][_jz_k], _jz_theNs[_jz_m]);
		 AssignJetToFloat(_jz_jet[0][_jz_m], _sjz_jet[0][_jz_m]);
		 /* state variable 1: */
		 DivideJetFloatA(_sjz_jet[1][_jz_m], _sjz_jet[8][_jz_k], _jz_theNs[_jz_m]);
		 AssignJetToFloat(_jz_jet[1][_jz_m], _sjz_jet[1][_jz_m]);
		 /* state variable 2: */
		 DivideJetFloatA(_sjz_jet[2][_jz_m], _sjz_jet[12][_jz_k], _jz_theNs[_jz_m]);
		 AssignJetToFloat(_jz_jet[2][_jz_m], _sjz_jet[2][_jz_m]);
		 _jz_initialized=1;
	 }
    _jz_lastOrder = order;
    _jz_ginitialized=1;
    if (sOut) {
     *sOut = _sjz_jet;
    }
    return(_jz_jet);
}

MY_FLOAT **taylor_coefficients_lorenz4_eqA(MY_FLOAT t, MY_FLOAT *x, int order, int rflag)
{
    return(taylor_coefficients_lorenz4_eq_A(t,x,order,rflag, NULL, NULL));
}

MY_FLOAT **taylor_coefficients_lorenz4_eq(MY_FLOAT t, MY_FLOAT *x, int order)
{
    return(taylor_coefficients_lorenz4_eqA(t,x,order,0));
}

#ifndef JET_IO_HELPERS
#define JET_IO_HELPERS


static const char *ode_variable_names[]={ "x", "y", "z", NULL };

static const char *jet_variable_monomials[]={

"s1","s2","s3",

"s1^2","s1s2","s1s3","s2^2","s2s3","s3^2",


"s1^3","s1^2s2","s1^2s3","s1s2^2","s1s2s3","s1s3^2","s2^3","s2^2s3","s2s3^2","s3^3",



"s1^4","s1^3s2","s1^3s3","s1^2s2^2","s1^2s2s3","s1^2s3^2","s1s2^3","s1s2^2s3","s1s2s3^2","s1s3^3","s2^4","s2^3s3","s2^2s3^2","s2s3^3","s3^4",




"s1^5","s1^4s2","s1^4s3","s1^3s2^2","s1^3s2s3","s1^3s3^2","s1^2s2^3","s1^2s2^2s3","s1^2s2s3^2","s1^2s3^3","s1s2^4","s1s2^3s3","s1s2^2s3^2","s1s2s3^3","s1s3^4","s2^5","s2^4s3","s2^3s3^2","s2^2s3^3","s2s3^4","s3^5",

NULL

};

#if _SIZE_OF_JET_VAR_ > 1

#include <stdio.h>
#include <ctype.h>
#include <string.h>

/*
 *  Jet IO Helpers.  These helpers assumes the jet libraries
 *  order the monomials using the lexicographical order. I.e, a
 *  jet is repesent as
 *    a0 + a1 s_1 + a2 s_2 + a3 s_3 + a4 s1^2 + a5 s1 s2 + a6 s1 s3 + a7 s2^2 + a8 s2 s3 + a9 s3^2 ...
 */

static char *get_next_number(const char *s, int *j, char *buf) {
  const char *p=s + *j;
  int i=0, k=0;
  while(p && *p && (!isdigit(*p)) && *p != '.' && *p != '-') {    k++; p++;  }
  while(p && *p && (isdigit(*p) || *p == '.' || *p == '-')) {  
    buf[i++] = *p++; k++;
  }
  buf[i++] = '\0';
  *j = *j+k;
  return buf;
}
static char *get_next_number_stdin(char *buf) {
  int i=0; char p = getc(stdin);
  while(p && (!isdigit(p)) && p != '.' && p != '-') { p = getc(stdin);  }
  while(p && (isdigit(p) || p == '.' || p == '-')) { buf[i++] = p; p = getc(stdin);}
  buf[i++] = '\0'; return buf;
}


static int taylor_gen_jetvar_prompt(int i, char *buf) {
  /* extern char *ode_variable_names[], *jet_variable_monomials[];  */
  
  const char *me = ode_variable_names[i];
  char tmp[128]; 
  int j;

  sprintf(buf, "%s", me);
  j = 0;
  while(jet_variable_monomials[j] != NULL) {
    sprintf(tmp, ",%s", jet_variable_monomials[j]);
    (void)strcat(buf, tmp);
    j++;
  }
  return j+1;
}  

/* 
 *  create jet var from an array of doubles
 *  Parms:  a, JET var
 *          values, an array of doubles assign to jet
 *                  caller needs to make sure there are  
 *                  enough number in values
 */

int taylor_make_jet(MY_JET a, MY_FLOAT *myfloats, double *values) {
  int i;
  MY_FLOAT *dtmp = (MY_FLOAT *)malloc((_size_of_jet_variable_) * sizeof(MY_FLOAT));
  if(dtmp == NULL) {
    fprintf(stderr, "Unable to allocate %d bytes.", (int) ( (_size_of_jet_variable_) * sizeof(MY_FLOAT)));
    exit(9);
  }  
  for(i = 0; i < _size_of_jet_variable_; i++) {
    InitMyFloat(dtmp[i]);
    if(myfloats != NULL) {
      AssignMyFloat( dtmp[i],myfloats[i]);
    } else if(values != NULL) {
      MakeMyFloatA(dtmp[i],values[i]);
    }
  }
  // The monomials solicited here are in lexical order. We need to
  // make sure our assigment macro/function are using the matching
  // order
  AssignFloatArrayToJet(a,dtmp);
  for(i = 0; i < _size_of_jet_variable_; i++) {
    ClearMyFloat(dtmp[i]);
  }
  (void)free(dtmp);
  return 0;
}

/*
 *  Make an identity jet. If values is give, set the value of the
 *  state variables. Otherwise, leave the state variable alone.
 */

int taylor_make_identity_jets(MY_JET *inOut, MY_FLOAT *myfloats, double *values) {
  int i,j, k;

  k = _number_of_jet_vars_ > _number_of_symbols_ ? _number_of_symbols_  : _number_of_jet_vars_;
  
  for(i = 0; i < k ; i++) {
    for(j = 0; j < _size_of_jet_variable_; j++) {
      if(j == 0 ) {
	if(myfloats != NULL) {
	  AssignMyFloat( MY_JET_DATA((inOut[i]),0),myfloats[i]);    	
	} else if( values != NULL) {
	    MakeMyFloatA( MY_JET_DATA((inOut[i]),0),values[i]);    	
	}
      } else {
	if(i == j-1) { // offset by 1.
	  MakeMyFloatA( MY_JET_DATA((inOut[i]),j),1.0);    		  
	} else {
	  MakeMyFloatA( MY_JET_DATA((inOut[i]),j),0.0);    		  
	}
      }
    }
  }
  for(i = k; i < _NUMBER_OF_STATE_VARS_; i++) {
    for(j = 0; j < _size_of_jet_variable_; j++) {
      if(j == 0 ) {
	if(myfloats != NULL) {
	  AssignMyFloat( MY_JET_DATA((inOut[i]),0),myfloats[i]);    	
	} else if( values != NULL) {	
	  MakeMyFloatA( MY_JET_DATA((inOut[i]),0),values[i]);    	
	}
      } else {
	MakeMyFloatA( MY_JET_DATA((inOut[i]),j),0.0);    		  
      }
    }
  }  
  return 0;
}

/*
 *   make an unit jet
 *     jet = v + s_idx
 *   idx is 0 based, s0, s1, s2 etc. 
 */
int taylor_make_unit_jet(MY_JET a, int idx, MY_FLOAT *myfloat, double *value) {
  for(int j = 0; j < _size_of_jet_variable_; j++) {
    if(j == 0) {
      if(myfloat != NULL) {
	AssignMyFloat( MY_JET_DATA((a),0),*myfloat);    		
      } else if(value != NULL) {
	MakeMyFloatA( MY_JET_DATA((a),0),*value);    		
      }
    } else {
      if(j == idx+1) {
	MakeMyFloatA( MY_JET_DATA((a),j),1.0);    		  	
      } else {
	MakeMyFloatA( MY_JET_DATA((a),j),0.0);
      }
    }
  }
  return 0;
}


/*
 *  Set the value of jet using an array of MY_FLOATs.
 *  optionally set the state variable. 
 */
int taylor_set_jet(MY_JET a, MY_FLOAT *myfloats, double *values, int include_state) {
  int j, offset = 1;
  if(include_state) offset = 0;
  else {MakeMyFloatA( MY_JET_DATA((a),0),0.0);}
  
  for(j = offset; j < _size_of_jet_variable_; j++) {
    if(myfloats != NULL) {
      AssignMyFloat( MY_JET_DATA((a),j),myfloats[j-offset]);
    } else if(values != NULL) {
      MakeMyFloatA( MY_JET_DATA((a),j),values[j-offset]);    		          
    } 
  }
  return 0;
}

/*
 *  Output a jet to an array of MY_FLOATs, optionally include
 *  the state variable.
 */
MY_FLOAT *taylor_convert_jet_to_array(MY_JET a, int include_state) {
  int i,offset = 1;
  if(include_state) offset = 0;
  
  static MY_FLOAT *dtmp = NULL;
#pragma omp threadprivate(dtmp)

  if(dtmp == NULL) {
    dtmp = (MY_FLOAT *)malloc((_size_of_jet_variable_) * sizeof(MY_FLOAT));
    if(dtmp == NULL) {
      fprintf(stderr, "Unable to allocate %d bytes.", (int) ( (_size_of_jet_variable_) * sizeof(MY_FLOAT)));
      exit(9);
    }  
    for(i = 0; i < _size_of_jet_variable_; i++) {
      InitMyFloat(dtmp[i]);
      MakeMyFloatA(dtmp[i], 0.0);
    }
  }
  for(i = offset; i < _size_of_jet_variable_; i++) {
    AssignMyFloat(dtmp[i-offset],  MY_JET_DATA((a),i));    
  }
  return dtmp;
}


/*
 *  taylor_input_jet: input jet from stdin,  prompt with list of monomials in jet var 
 *
 *  Parms:  a, JET var
 *          idx, index of a in Jet Array. The order
 *             of the Jet Array is affixed with the
 *             order of LHS of ODEs 
 *          values, an array of doubles assign to jet
 *                  caller needs to make sure there are  
 *                  enough number in values
 */

int taylor_input_jet_from_stdin(MY_JET a, int idx) {
  /* extern char *ode_variable_names[]; */
  char buf[2048];
  int i, count, nbytes;

  MY_FLOAT *dtmp = (MY_FLOAT *)malloc((_size_of_jet_variable_) * sizeof(MY_FLOAT));
  if(dtmp == NULL) {
    fprintf(stderr, "Unable to allocate %d bytes.", (int) ( (_size_of_jet_variable_) * sizeof(MY_FLOAT)));
    exit(9);
  }  
  for(i = 0; i < _size_of_jet_variable_; i++) {
    InitMyFloat(dtmp[i]);
  }
  
  bzero(buf, 2048);
  count = taylor_gen_jetvar_prompt(idx, buf);

  fprintf(stderr, "Enter values for JET Var %s: %s \n", ode_variable_names[idx], buf);
  buf[0] =0;
  for(i = 0; i < _size_of_jet_variable_; i++) {
    char *q = get_next_number_stdin(buf);
    double f = atof(q);
    MakeMyFloatC(dtmp[i], buf, f);    
  }
  
  taylor_make_jet(a,dtmp, NULL);
  
  for(i = 0; i < _size_of_jet_variable_; i++) {
    ClearMyFloat(dtmp[i]);
  }  
  (void) free(dtmp);

  return 0;
}

int taylor_input_jet_from_string(MY_JET a, const char *str) {
  static char buf[256];
  static MY_FLOAT *dtmp = NULL;  
#pragma omp threadprivate(buf,dtmp)
  int i, j=0;
  if(dtmp == NULL) {
    dtmp = (MY_FLOAT *)malloc((_size_of_jet_variable_) * sizeof(MY_FLOAT));
    if(dtmp == NULL) {
      fprintf(stderr, "Unable to allocate %d bytes.", (int) ( (_size_of_jet_variable_) * sizeof(MY_FLOAT)));
      exit(9);
    }  
    for(i = 0; i < _size_of_jet_variable_; i++) {
      InitMyFloat(dtmp[i]);
      MakeMyFloatA(dtmp[i], 0.0);
    }
  }
  for(i=0; i< _size_of_jet_variable_; i++) {
    get_next_number(str, &j, buf);
    double f=atof(buf);
    MakeMyFloatC(dtmp[i], buf, f);
  }

  taylor_make_jet(a, dtmp, NULL);
  return 0;
}

int taylor_output_jet(FILE *file, char *fmt, MY_JET a) {
  int i;
  static MY_FLOAT *dtmp = NULL;
#pragma omp threadprivate(dtmp)

  if(dtmp == NULL) {
    dtmp = (MY_FLOAT *)malloc((_size_of_jet_variable_) * sizeof(MY_FLOAT));
    if(dtmp == NULL) {
      fprintf(stderr, "Unable to allocate %d bytes.", (int) ( (_size_of_jet_variable_) * sizeof(MY_FLOAT)));
      exit(9);
    }  
    for(i = 0; i < _size_of_jet_variable_; i++) {
      InitMyFloat(dtmp[i]);
      MakeMyFloatA(dtmp[i], 0.0);
    }
  }
  //  for(i = 0; i < _size_of_jet_variable_; i++) {
  //    MakeMyFloatA(dtmp[i], 0.0);    
  //  }

  AssignJetToFloatArray(dtmp,a);  
  
  for(i = 0; i < _size_of_jet_variable_; i++) {
    OutputMyFloat3(file, fmt, dtmp[i]);
    //#if defined _USE_MPFR_ || defined  _USE_GMP_
    //    fprintf(file, " ");
    //#endif
  }
  return 0;  
}

void taylor_initialize_jet_library2(int nvars, int degree) {
  InitUpJet2(nvars, degree);
}
void taylor_initialize_jet_library(void) {
  InitUpJet();
}

void taylor_initialize_jet_variable(MY_JET *jet) {
  InitJet(*jet);
}

void taylor_clear_jet_variable(MY_JET *jet) {
  ClearJet(*jet);
}
void taylor_clear_up_jet_library(void) {
  ClearUpJet();
}

int taylor_set_jet_variable_degree(int deg) {
  return SetJetVarDegree(deg);
}
int taylor_set_jet_variable_number_of_symbols(int nsyms) {
  return SetNumSymbols(nsyms);
}

const char **taylor_get_variable_names() {
  return ode_variable_names;
}

const char **taylor_get_jet_monomials() {
  return jet_variable_monomials;
}
#endif




static MY_FLOAT *taylor_jet_monomial_values(double *params) {
	MY_FLOAT *monomial_values = (MY_FLOAT *)malloc( (57) * sizeof(MY_FLOAT));
	MY_FLOAT ftmp, ptmp, qtmp, one;
	int i;

	for(i=0;i<57;i++) {;
		InitMyFloat(monomial_values[i]);
	}
	InitMyFloat(ftmp);
	InitMyFloat(ptmp);
	InitMyFloat(qtmp);
	InitMyFloat(one);
	MakeMyFloatA(one,1);
	/* "s1" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	AssignMyFloat(monomial_values[1], ftmp);

	/* "s2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	AssignMyFloat(monomial_values[2], ftmp);

	/* "s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[3], ftmp);

	/* "s1^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[4], ftmp);

	/* "s1s2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	AssignMyFloat(monomial_values[5], ftmp);

	/* "s1s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[6], ftmp);

	/* "s2^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[7], ftmp);

	/* "s2s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[8], ftmp);

	/* "s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[9], ftmp);

	/* "s1^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[10], ftmp);

	/* "s1^2s2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	AssignMyFloat(monomial_values[11], ftmp);

	/* "s1^2s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[12], ftmp);

	/* "s1s2^2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[13], ftmp);

	/* "s1s2s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[14], ftmp);

	/* "s1s3^2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[15], ftmp);

	/* "s2^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[16], ftmp);

	/* "s2^2s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[17], ftmp);

	/* "s2s3^2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[18], ftmp);

	/* "s3^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[19], ftmp);

	/* "s1^4" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[20], ftmp);

	/* "s1^3s2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	AssignMyFloat(monomial_values[21], ftmp);

	/* "s1^3s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[22], ftmp);

	/* "s1^2s2^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[23], ftmp);

	/* "s1^2s2s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[24], ftmp);

	/* "s1^2s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[25], ftmp);

	/* "s1s2^3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[26], ftmp);

	/* "s1s2^2s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[27], ftmp);

	/* "s1s2s3^2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[28], ftmp);

	/* "s1s3^3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[29], ftmp);

	/* "s2^4" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[30], ftmp);

	/* "s2^3s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[31], ftmp);

	/* "s2^2s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[32], ftmp);

	/* "s2s3^3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[33], ftmp);

	/* "s3^4" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[34], ftmp);

	/* "s1^5" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(5));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[35], ftmp);

	/* "s1^4s2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	AssignMyFloat(monomial_values[36], ftmp);

	/* "s1^4s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[37], ftmp);

	/* "s1^3s2^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[38], ftmp);

	/* "s1^3s2s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[39], ftmp);

	/* "s1^3s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[40], ftmp);

	/* "s1^2s2^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[41], ftmp);

	/* "s1^2s2^2s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[42], ftmp);

	/* "s1^2s2s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[43], ftmp);

	/* "s1^2s3^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[0]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[44], ftmp);

	/* "s1s2^4" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[45], ftmp);

	/* "s1s2^3s3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[46], ftmp);

	/* "s1s2^2s3^2" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[47], ftmp);

	/* "s1s2s3^3" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[48], ftmp);

	/* "s1s3^4" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[0]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[49], ftmp);

	/* "s2^5" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(5));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[50], ftmp);

	/* "s2^4s3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MultiplyMyFloatD(ftmp,  ftmp,  params[2]);
	AssignMyFloat(monomial_values[51], ftmp);

	/* "s2^3s3^2" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[52], ftmp);

	/* "s2^2s3^3" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[1]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(2));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(3));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[53], ftmp);

	/* "s2s3^4" */
	AssignMyFloat(ftmp, one);
	MultiplyMyFloatD(ftmp,  ftmp,  params[1]);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(4));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[54], ftmp);

	/* "s3^5" */
	AssignMyFloat(ftmp, one);
	MakeMyFloatA(ptmp,  params[2]);
	ExponentiateMyFloatIA(qtmp, ptmp, (double)(5));;
	MultiplyMyFloatA(ftmp, ftmp, qtmp);
	AssignMyFloat(monomial_values[55], ftmp);

	ClearMyFloat(ftmp);
	ClearMyFloat(ptmp);
	ClearMyFloat(qtmp);
	ClearMyFloat(one);

	return monomial_values;
}

void taylor_jet_reduce(MY_JET a, double *params) {
	MY_FLOAT *monomials = taylor_jet_monomial_values(params);
	static int inited=0;
	static MY_FLOAT ftmp;
#pragma omp threadprivate(inited,ftmp)
	int i;

	if(inited==0){ inited=1;InitMyFloat(ftmp); }
	for(i=1; i< 56; i++) {
		MultiplyMyFloatA(ftmp, MY_JET_DATA((a),i),monomials[i]);
		AddMyFloatA(MY_JET_DATA((a),0), MY_JET_DATA((a),0), ftmp);
	};
	for(i=0; i< 57; i++) {ClearMyFloat(monomials[i]);}
	free(monomials);
}
#endif

