/*
this file contains the routines used to write a realified (but still
stored in complex variables) change of variables in real form into an
ascii or binary file. it is assumed that the change corresponds to a
reduction to the central manifold (the first two variables correspond
to the hyperbolic directions and the last four variables correspond to
the central directions). hence, the change to be written depends on 6
variables and it is handled by mp6s or mp6p, while the written change
depends on 4 variables (the hyperbolic variables are taken zero) and
it is handled by mp4s and mp4p. it is also assumed that all these
manipulators have been initialized.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

extern "C" {
#include "mp4s.h"
#include "mp4p.h"
#include "mp6s.h"
#include "mp6p.h"
#include "msgs.h"
}

#include "arit-c.h"

void wrcva4s(complex **c, integer n, char *nf1, char *nf2, double tol)
/*
this routine writes (in an ascii file) the realified change of
variables stored in c (according to mp6s). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one of the positions. then, the routine takes the real and imaginary
part of c (and scales them by sqrt(2) and -sqrt(2) resp.) to obtain
the real change corresponding to the same position and the related
momentum. this change is written to a file as a mp4s expansion (the
first two variables have been taken equal to zero).

parameters:
c:   change of variables corresponding to one of the variables (input).
n:   degree of c.
nf1: name of the file to store the realified change for the position
     (input).
nf2: name of the file to store the realified change for the momentum
     (input).
tol: threshold. coefficients that are smaller than tol are not written.
*/
{
   renum r2,t;
   integer i,j,k[6],m;
   char *fmt;
   FILE *f1,*f2;
   r2=sqrt(2);
   fmt="%2d %2d %2d %2d %24.16e\n";
   f1=fopen(nf1,"w");
   if (f1 == NULL) {printf("wrcva4s: %s can not be opened\n",nf1); exit(1);}
   f2=fopen(nf2,"w");
   if (f2 == NULL) {printf("wrcva4s: %s can not be opened\n",nf2); exit(1);}
   for (i=1; i<=n; i++)
   {
      m=ntph6s(i);
      for (j=0; j<m; j++)
      {
         llex6s(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         t=r2*real(c[i][j]);
         if (fabs(t) > tol) fprintf(f1,fmt,k[2],k[3],k[4],k[5],t);
         t=-r2*imag(c[i][j]);
         if (fabs(t) > tol) fprintf(f2,fmt,k[2],k[3],k[4],k[5],t);
      }
   }
   fclose(f2);
   fclose(f1);
}
void wrcva4p(complex **c, integer n, char *nf1, char *nf2, double tol)
/*
this routine writes (in an ascii file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one of the positions. then, the routine takes the real and imaginary
part of c (and scales them by sqrt(2) and -sqrt(2) resp.) to obtain
the real change corresponding to the same position and the related
momentum. this change is written to a file as a mp4p expansion (the
first two variables have been taken equal to zero).

parameters:
c:   change of variables corresponding to one of the variables (input).
n:   degree of c.
nf1: name of the file to store the realified change for the position
     (input).
nf2: name of the file to store the realified change for the momentum
     (input).
tol: threshold. coefficients that are smaller than tol are not written.
*/
{
   renum r2,t;
   integer i,j,k[6],m;
   char *fmt;
   FILE *f1,*f2;
   r2=sqrt(2);
   fmt="%2d %2d %2d %2d %24.16e\n";
   f1=fopen(nf1,"w");
   if (f1 == NULL) {printf("wrcva4p: %s can not be opened\n",nf1); exit(1);}
   f2=fopen(nf2,"w");
   if (f2 == NULL) {printf("wrcva4p: %s can not be opened\n",nf2); exit(1);}
   for (i=1; i<=n; i++)
   {
      m=ntph6p(i);
      for (j=0; j<m; j++)
      {
         llex6p(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         t=r2*real(c[i][j]);
         if (fabs(t) > tol) fprintf(f1,fmt,k[2],k[3],k[4],k[5],t);
         t=-r2*imag(c[i][j]);
         if (fabs(t) > tol) fprintf(f2,fmt,k[2],k[3],k[4],k[5],t);
      }
   }
   fclose(f2);
   fclose(f1);
}
void wrcvb4s(complex **c, integer n, char *nf1, char *nf2)
/*
this routine writes (in a binary file) the realified change of
variables stored in c (according to mp6s). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one of the positions. then, the routine takes the real and imaginary
part of c (and scales them by sqrt(2) and -sqrt(2) resp.) to obtain
the real change corresponding to the same position and the related
momentum. this change is written to a file as a mp4s expansion (the
first two variables have been taken equal to zero).

parameters:
c:   change of variables corresponding to one of the variables (input).
n:   degree of c.
nf1: name of the file to store the realified change for the position
     (input).
nf2: name of the file to store the realified change for the momentum
     (input).
*/
{
   renum r2,*tr,*ti;
   integer i,j,k[6],kk[4],l,m,nt,inf[4];
   FILE *f1,*f2;
   r2=sqrt(2);
   f1=fopen(nf1,"wb");
   if (f1 == NULL) {printf("wrcvb4s: %s can not be opened\n",nf1); exit(1);}
   inf[0]=4;  /* number of variables */
   inf[1]=2;  /* kind of symmetry */
   inf[2]=1; /* initial degree */
   inf[3]=n; /* final degree */
   j=fwrite(inf,sizeof(integer),4,f1);
   if (j < 4) iomsg_eb2("wrcvb4s",nf1); /* this will stop the program */
   f2=fopen(nf2,"wb");
   if (f2 == NULL) {printf("wrcvb4s: %s can not be opened\n",nf2); exit(1);}
   j=fwrite(inf,sizeof(integer),4,f2);
   if (j < 4) iomsg_eb2("wrcvb4s",nf2); /* this will stop the program */
   nt=ntph4s(n);
   tr=(renum*)malloc(nt*sizeof(renum));
   if (tr == NULL) {puts("wrcvb4s: out of memory (1)"); exit(1);}
   ti=(renum*)malloc(nt*sizeof(renum));
   if (ti == NULL) {puts("wrcvb4s: out of memory (2)"); exit(1);}
   for (i=1; i<=n; i++)
   {
      nt=ntph4s(i);
      for (j=0; j<nt; j++) tr[j]=0;
      for (j=0; j<nt; j++) ti[j]=0;
      m=ntph6s(i);
      for (j=0; j<m; j++)
      {
         llex6s(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         kk[0]=k[2]; kk[1]=k[3]; kk[2]=k[4]; kk[3]=k[5];
         l=exll4s(kk,i);
         tr[l]=r2*real(c[i][j]);
         ti[l]=-r2*imag(c[i][j]);
      }
      fwrite(tr,sizeof(renum),nt,f1);
      fwrite(ti,sizeof(renum),nt,f2);
   }
   free(ti); free(tr);
   fclose(f2);
   fclose(f1);
}
void wrcvb4p(complex **c, integer n, char *nf1, char *nf2)
/*
this routine writes (in a binary file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one of the positions. then, the routine takes the real and imaginary
part of c (and scales them by sqrt(2) and -sqrt(2) resp.) to obtain
the real change corresponding to the same position and the related
momentum. this change is written to a file as a mp4p expansion (the
first two variables have been taken equal to zero).

parameters:
c:   change of variables corresponding to one of the variables (input).
n:   degree of c.
nf1: name of the file to store the realified change for the position
     (input).
nf2: name of the file to store the realified change for the momentum
     (input).
*/
{
   renum r2,*tr,*ti;
   integer i,j,k[6],kk[4],l,m,nt,inf[4];
   FILE *f1,*f2;
   r2=sqrt(2);
   f1=fopen(nf1,"wb");
   if (f1 == NULL) {printf("wrcvb4p: %s can not be opened\n",nf1); exit(1);}
   inf[0]=4;  /* number of variables */
   inf[1]=1;  /* kind of symmetry */
   inf[2]=1; /* initial degree */
   inf[3]=n; /* final degree */
   j=fwrite(inf,sizeof(integer),4,f1);
   if (j < 4) iomsg_eb2("wrcvb4p",nf1); /* this will stop the program */
   f2=fopen(nf2,"wb");
   if (f2 == NULL) {printf("wrcvb4p: %s can not be opened\n",nf2); exit(1);}
   j=fwrite(inf,sizeof(integer),4,f2);
   if (j < 4) iomsg_eb2("wrcvb4p",nf2); /* this will stop the program */
   nt=ntph4p(n);
   tr=(renum*)malloc(nt*sizeof(renum));
   if (tr == NULL) {puts("wrcvb4p: out of memory (1)"); exit(1);}
   ti=(renum*)malloc(nt*sizeof(renum));
   if (ti == NULL) {puts("wrcvb4p: out of memory (2)"); exit(1);}
   for (i=1; i<=n; i++)
   {
      nt=ntph4p(i);
      for (j=0; j<nt; j++) tr[j]=0;
      for (j=0; j<nt; j++) ti[j]=0;
      m=ntph6p(i);
      for (j=0; j<m; j++)
      {
         llex6p(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         kk[0]=k[2]; kk[1]=k[3]; kk[2]=k[4]; kk[3]=k[5];
         l=exll4p(kk,i);
         tr[l]=r2*real(c[i][j]);
         ti[l]=-r2*imag(c[i][j]);
      }
      fwrite(tr,sizeof(renum),nt,f1);
      fwrite(ti,sizeof(renum),nt,f2);
   }
   free(ti); free(tr);
   fclose(f2);
   fclose(f1);
}
void wrrva4s(complex **c, integer n, char *nf, double tol)
/*
this routine writes (in an ascii file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one variable that has not been complexified. the routine takes the
real part of c and writes it to a file, according to the mp4s format.

parameters:
c:   change of variables corresponding to one of the variables
n:   degree of c.
nf:  name of the file to store the realified change (input).
tol: threshold to check if the imaginary parts are zero. if some of
     them is bigger than tol, a warning message is issued. only
     coefficients bigger than tol are written.
*/
{
   renum t;
   integer i,j,k[6],m;
   char *fm1,*fm2;
   FILE *fi;
   fm1="%2d %2d %2d %2d %24.16e\n";
   fm2="wrrva4s: imag too big. %2d %2d %2d %2d %2d %2d %24.16e\n";
   fi=fopen(nf,"w");
   if (fi == NULL) {printf("wrrva4s: %s can not be opened\n",nf); exit(1);}
   for (i=1; i<=n; i++)
   {
      m=ntph6s(i);
      for (j=0; j<m; j++)
      {
         llex6s(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         t=real(c[i][j]);
         if (fabs(t) > tol) fprintf(fi,fm1,k[2],k[3],k[4],k[5],t);
         t=fabs(imag(c[i][j]));
         if (t > tol) printf(fm2,k[0],k[1],k[2],k[3],k[4],k[5],t);
      }
   }
   fclose(fi);
}
void wrrva4p(complex **c, integer n, char *nf, double tol)
/*
this routine writes (in an ascii file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one variable that has not been complexified. the routine takes the
real part of c ans writes it to a file, according to the mp4p format.

parameters:
c:   change of variables corresponding to one of the variables
n:   degree of c.
nf:  name of the file to store the realified change (input).
tol: threshold to check if the imaginary parts are zero. if some of
     them is bigger than tol, a warning message is issued. only
     coefficients bigger than tol are written.
*/
{
   renum t;
   integer i,j,k[6],m;
   char *fm1,*fm2;
   FILE *fi;
   fm1="%2d %2d %2d %2d %24.16e\n";
   fm2="wrrva4p: imag too big. %2d %2d %2d %2d %2d %2d %24.16e\n";
   fi=fopen(nf,"w");
   if (fi == NULL) {printf("wrrva4p: %s can not be opened\n",nf); exit(1);}
   for (i=1; i<=n; i++)
   {
      m=ntph6p(i);
      for (j=0; j<m; j++)
      {
         llex6p(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         t=real(c[i][j]);
         if (fabs(t) > tol) fprintf(fi,fm1,k[2],k[3],k[4],k[5],t);
         t=fabs(imag(c[i][j]));
         if (t > tol) printf(fm2,k[0],k[1],k[2],k[3],k[4],k[5],t);
      }
   }
   fclose(fi);
}
void wrrvb4s(complex **c, integer n, char *nf)
/*
this routine writes (in a binary file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one variable that has not been complexified. the routine takes the
real part of c and writes it to a file, according to the mp4s format.

parameters:
c:   change of variables corresponding to one of the variables
n:   degree of c.
nf:  name of the file to store the realified change (input).
*/
{
   renum *t;
   integer i,j,k[6],kk[4],l,m,nt,inf[4];
   FILE *fi;
   fi=fopen(nf,"wb");
   if (fi == NULL) {printf("wrrvb4s: %s can not be opened\n",nf); exit(1);}
   inf[0]=4;  /* number of variables */
   inf[1]=2;  /* kind of symmetry */
   inf[2]=1; /* initial degree */
   inf[3]=n; /* final degree */
   j=fwrite(inf,sizeof(integer),4,fi);
   if (j < 4) iomsg_eb2("wrrvb4s",nf); /* this will stop the program */
   nt=ntph4s(n);
   t=(renum*)malloc(nt*sizeof(renum));
   if (t == NULL) {puts("wrrvb4s: out of memory"); exit(1);}
   for (i=1; i<=n; i++)
   {
      nt=ntph4s(i);
      for (j=0; j<nt; j++) t[j]=0;
      m=ntph6s(i);
      for (j=0; j<m; j++)
      {
         llex6s(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         kk[0]=k[2]; kk[1]=k[3]; kk[2]=k[4]; kk[3]=k[5];
         l=exll4s(kk,i);
         t[l]=real(c[i][j]);
      }
      fwrite(t,sizeof(renum),nt,fi);
   }
   free(t);
   fclose(fi);
}
void wrrvb4p(complex **c, integer n, char *nf)
/*
this routine writes (in a binary file) the realified change of
variables stored in c (according to mp6p). it is assumed that c is the
result of inserting the realifying change into the (complex) change of
one variable that has not been complexified. the routine takes the
real part of c and writes it to a file, according to the mp4p format.

parameters:
c:   change of variables corresponding to one of the variables
n:   degree of c.
nf:  name of the file to store the realified change (input).
*/
{
   renum *t;
   integer i,j,k[6],kk[4],l,m,nt,inf[4];
   FILE *fi;
   fi=fopen(nf,"wb");
   if (fi == NULL) {printf("wrrvb4p: %s can not be opened\n",nf); exit(1);}
   inf[0]=4;  /* number of variables */
   inf[1]=1;  /* kind of symmetry */
   inf[2]=1; /* initial degree */
   inf[3]=n; /* final degree */
   j=fwrite(inf,sizeof(integer),4,fi);
   if (j < 4) iomsg_eb2("wrrvb4p",nf); /* this will stop the program */
   nt=ntph4p(n);
   t=(renum*)malloc(nt*sizeof(renum));
   if (t == NULL) {puts("wrrvb4p: out of memory"); exit(1);}
   for (i=1; i<=n; i++)
   {
      nt=ntph4p(i);
      for (j=0; j<nt; j++) t[j]=0;
      m=ntph6p(i);
      for (j=0; j<m; j++)
      {
         llex6p(j,k,i);
         if ((k[0] != 0) || (k[1] != 0)) continue;
         kk[0]=k[2]; kk[1]=k[3]; kk[2]=k[4]; kk[3]=k[5];
         l=exll4p(kk,i);
         t[l]=real(c[i][j]);
      }
      fwrite(t,sizeof(renum),nt,fi);
   }
   free(t);
   fclose(fi);
}
