/*
this file contains the routines that realify a 6-variables power
expansion (see file mp6p.c to see the kind of expansions handled).
before using these routines, a call to ini_real (file irex.cc) must
be made to compute the table cm2 (see below and file irex.cc).
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "arit-c.h"

extern "C" {
#include "mp3.h"
#include "mp6p.h"
#include "io6p.h"
}

#include "irex.h"

void re6p(integer rf[3], complex **f, integer n, complex **cm2, char *tmp)
/*
this is to realify the (complex) expansion contained in f. the
result is overwritten in the same space f.

parameters:
rf:  realification flags. the couple (conjugated variables) number i
     is realified only if rf[i] is 1 (input).
f:   on input:  expansion to be realified.
     on output: realified expansion. the imaginary parts must be zero
                (but, of course, they will not be exactly zero due to
                the propagation of errors).
n:   degree of f.
cm2: working table. it is computed by ini_real (see file irex.cc)
     (input).
tmp: temporary file, to be used as working space. the file will be
     created and erased by the routine.
*/
{
   void remo6p(integer rf[3], integer k[6], complex f, complex *p,
               complex **cm2);
   complex c;
   integer i,j,r,m,k[6],inf[4];
   FILE *file;
   puts("re6p: realifying...");
   web6p(f,1,n,tmp);
   for (r=0; r<=n; r++)
   {
      m=ntph6p(r);
      for (j=0; j<m; j++) f[r][j]=0;
   }
   file=fopen(tmp,"rb");
   if (file == NULL)
      {
         printf("re6p error: can't open the file %s\n",tmp);
         puts("(this is an internal temporary file)");
         exit(1);
      }
   fread(inf,sizeof(integer),4,file); /* this is the head of the file */
   for (r=1; r<=n; r++)
   {
      printf("degree %d\n",r);
      m=ntph6p(r);
      for (j=0; j<m; j++)
      {
         llex6p(j,k,r);
         i=fread(&c,sizeof(complex),1,file);
         if (i != 1) {printf("re6p error. i: %d\n",i); exit(1);}
         remo6p(rf,k,c,f[r],cm2);
      }
   }
   fclose(file);
   j=remove(tmp);
   if (j != 0) printf("re6p warning: cannot remove %s\n",tmp);
   puts("re6p: end of the realification.");
   return;
}
void remo6p(integer rf[3], integer k[6], complex f, complex *p, complex **cm2)
/*
this is to realify the monomial of exponent k. it is assumed that
k is such that k[4]+k[5] is odd.
the algorithm is the following: each couple position-momentum produces
an homogeneous polynomial (with the same degree as the couple) when
is realified. the coefficients of these polynomials are stored
in the suitable places of cm2. hence, we will use three nested loops
(one for each couple) to multiply these three homogeneous polynomials.
of course, we will do the product at the same time we compute each one
of the coefficients.

parameters:
rf:  realification flags. the couple (conjugated variables) number i
     is realified only if rf[i] is 1 (input).
k:   exponent of the monomial to be realified (input).
f:   coefficient of the monomial.
p:   homogeneous polynomial of degree k[0]+...+k[5]. the result of the
     realification is added to the values contained in p. although
     this polynomial can be taken real, we prefer to use a complex
     one and, when the realification is finished, to check that the
     imaginary parts of p vanish (input/output).
cm2: working table. see routine ini_real for details (input).
*/
{
   complex w,w1,w2,w3;
   integer n,n1,n2,n3,l[6],lloc,i1,i2,i3,m[3],j1,j2,j3;
   n1=k[0]+k[1];
   n2=k[2]+k[3];
   n3=k[4]+k[5];
   n=n1+n2+n3;
/*
   next line is the loop that runs over the coefficients of the
   homogeneous polynomial obtained from the realification of the
   monomial (k[0],k[1])
*/
   for (i1=0; i1<=n1; i1++)
   {
/*
      now we compute the corresponding coefficient w1
*/
      if (rf[0] == 1)
         {
            m[0]=k[0];
            m[1]=k[1];
            m[2]=i1;
            j1=n1+i1;
            lloc=exll3(m,j1);
            w1=cm2[j1][lloc];
/*
            l[0] and l[1] refer to the exponents of the realified variables.
*/
            l[0]=n1-i1;
            l[1]=i1;
         }
         else
         {
            w1=1;
            l[0]=k[0];
            l[1]=k[1];
            i1=n1;
         }
/*
      in the next two loops we repeat the same computation but for
      the other variables
*/
      for (i2=0; i2<=n2; i2++)
      {
         if (rf[1] == 1)
            {
               m[0]=k[2];
               m[1]=k[3];
               m[2]=i2;
               j2=n2+i2;
               lloc=exll3(m,j2);
               w2=cm2[j2][lloc];
               l[2]=n2-i2;
               l[3]=i2;
            }
            else
            {
               w2=1;
               l[2]=k[2];
               l[3]=k[3];
               i2=n2;
            }
         for (i3=0; i3<=n3; i3++)
         {
            if (rf[2] == 1)
               {
                  m[0]=k[4];
                  m[1]=k[5];
                  m[2]=i3;
                  j3=n3+i3;
                  lloc=exll3(m,j3);
                  w3=cm2[j3][lloc];
                  l[4]=n3-i3;
                  l[5]=i3;
               }
               else
               {
                  w3=1;
                  l[4]=k[4];
                  l[5]=k[5];
                  i3=n3;
               }
/*
            now, l[0],...,l[5] contain the corresponding exponent in the
            realified variables. the coefficient of this monomial is
            given by f*w1*w2*w3.
*/
            lloc=exll6p(l,n);
            w=w1*w2*w3;
            p[lloc] += w*f;
         }
      }
   }
   return;
}
