/*
this file contains the routines that reduce the hamiltonian (handled
as a power expansion by the routines in mp6s.c) to its central part.
it is assumed that the hamiltonian is in (still complex) partial
normal form, but with the monomials that "couple" the central and
hyperbolic directions killed. What these routines do is to get a 4
variables power expansion (handled by mp4s) by skipping the monomials
that vanish on the central manifold.  before using these routines, a
call to ini_real (file irex.cc) must be made to compute the table cm2
(see below and file irex.cc).
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "arit-c.h"

extern "C" {
#include "mp3.h"
#include "mp4s.h"
#include "mp6s.h"
#include "io6s.h"
#include "msgs.h"
#include "kill-cm.h"
}

#include "irex.h"

void rcm6s(integer rf[3], complex **f, integer n, complex **cm2, char *tmp)
/*
this routine computes the reduction to the central manifold for the
hamiltonian (handled by mp6s) stored in the binary file tmp. It is
assumed that this hamiltonian is in complex partial normal form, and
that the two first variables correspond to the hyperbolic directions
and that the four remaining ones correspond to the central manifold.
then, this routine realifies the hamiltonian and, at the same time,
only keeps the monomials corresponding to the central manifold.

parameters:
rf:  realification flags. the couple (conjugated variables) number i
     is realified only if rf[i] is 1. usually, this array must
     contain the values (0,1,1), but different values can be used
     in different contexts (input).
f:   real expansion of the central manifold. on input, this is a
     pointer to an (empty) 4 variables expansion (handled by
     mp4s.c). on output, this space is overwritten by the central
     manifold. the imaginary parts must be zero (but, of course, they
     will not be exactly zero due to the propagation of numerical
     errors) (output).
n:   degree up to which the central manifold is wanted.
cm2: working table. it is computed by ini_real (see file irex.cc)
     (input).
tmp: name of the file containing the expansion. this file is in binary
     form, and it has been written by routine web6s (see file io6s.c).
     the initial degree of the expansion must be 2 (input).
*/
{
   void remo4s(integer rf[3], integer k[6], complex f, complex *p,
               complex **cm2);
   complex c;
   float a,e;        /* these variables are only used in a check */
   integer i,j,r,m,k[6],inf[4];
   FILE *file;
   puts("rcm6s: starting realification...");
   for (r=0; r<=n; r++)
   {
      m=ntph4s(r);
      for (j=0; j<m; j++) f[r][j]=0;
   }
   file=fopen(tmp,"rb");
   if (file == NULL) {puts("rcm6s: error 1."); exit(1);}
   j=fread(inf,sizeof(integer),4,file); /* this is the head of the file */
   if (j < 4) {puts("rcm6s: error 2."); exit(1);}
   if ((inf[0] != 6) || (inf[1] != 2)) iomsg_eb02("rcm",tmp,inf[0],inf[1]);
   iomsg_eb01("rcm6s",tmp,2,n,inf[2],inf[3]);
   for (r=2; r<=n; r++)
   {
      e=0;
      m=ntph6s(r);
      for (j=0; j<m; j++)
      {
         llex6s(j,k,r);
         i=fread(&c,sizeof(complex),1,file);
         if (i != 1) {printf("rcm6s error 3. i: %d\n",i); exit(1);}
         a=sqrt(real(c)*real(c)+imag(c)*imag(c));
         if (kill_cm(k) == 1) {e=((a>e)? a: e); continue;}
         if (k[0]+k[1] > 1) continue;
         remo4s(rf,k,c,f[r],cm2);
      }
      printf("rcm6s: degree %3d:   e: %e\n",r,e);
   }
   fclose(file);
   puts("rcm6s: realification ended");
   return;
}
void remo4s(integer rf[3], integer k[6], complex f, complex *p, complex **cm2)
/*
this is to realify the monomial of exponent k. it is assumed that
k[0]=k[1]=0 and that k[4]+k[5] is even.
the algorithm is the following: each couple position-momentum produces
an homogeneous polynomial (with the same degree as the couple) when
is realified. the coefficients of these polynomials are stored
in the suitable places of cm2. hence, we will use three nested loops
(one for each couple) to multiply these three homogeneous polynomials.
of course, we will do the product at the same time we compute each one
of the coefficients.

parameters:
rf:  realification flags. the couple (conjugated variables) number i
     is realified only if rf[i] is 1 (input).
k:   exponent of the monomial to be realified (input).
f:   coefficient of the monomial to be realified.
p:   homogeneous polynomial of degree k[2]+...+k[5]. the result of the
     realification is added to the values contained in p. although
     this polynomial is real, we prefer to store it in a complex
     one and, when the realification is finished, to check that the
     imaginary parts of p vanish (input/output).
cm2: working table. see routine ini_real for details (input).
*/
{
   complex w,w1,w2,w3;
   integer n,n1,n2,n3,l[6],lloc,i1,i2,i3,m[3],j1,j2,j3,ll[4];
   char *fmt;
   fmt="remo4s error: monomial %d %d %d %d %d %d\n";
   n1=k[0]+k[1];
   n2=k[2]+k[3];
   n3=k[4]+k[5];
   n=n1+n2+n3;
   if (n1 != 0)
   {
      printf(fmt,k[0],k[1],k[2],k[3],k[4],k[5]);
      exit(1);
   }
/*
   next line is the loop that runs over the coefficients of the
   homogeneous polynomial obtained from the realification of the
   monomial (k[0],k[1])
*/
   for (i1=0; i1<=n1; i1++)
   {
/*
      now we compute the corresponding coefficient w1
*/
      if (rf[0] == 1)
         {
            m[0]=k[0];
            m[1]=k[1];
            m[2]=i1;
            j1=n1+i1;
            lloc=exll3(m,j1);
            w1=cm2[j1][lloc];
/*
            l[0] and l[1] refer to the exponents of the realified variables.
*/
            l[0]=n1-i1;
            l[1]=i1;
         }
         else
         {
            w1=1;
            l[0]=k[0];
            l[1]=k[1];
         }
/*
      in the next two loops we repeat the same computation but for
      the other variables
*/
      for (i2=0; i2<=n2; i2++)
      {
         if (rf[1] == 1)
            {
               m[0]=k[2];
               m[1]=k[3];
               m[2]=i2;
               j2=n2+i2;
               lloc=exll3(m,j2);
               w2=cm2[j2][lloc];
               l[2]=n2-i2;
               l[3]=i2;
            }
            else
            {
               w2=1;
               l[2]=k[2];
               l[3]=k[3];
            }
         for (i3=0; i3<=n3; i3++)
         {
            if (rf[2] == 1)
               {
                  m[0]=k[4];
                  m[1]=k[5];
                  m[2]=i3;
                  j3=n3+i3;
                  lloc=exll3(m,j3);
                  w3=cm2[j3][lloc];
                  l[4]=n3-i3;
                  l[5]=i3;
               }
               else
               {
                  w3=1;
                  l[4]=k[4];
                  l[5]=k[5];
               }
/*
            now, l[0],...,l[5] contain the corresponding exponent in the
            realified variables. the coefficient of this monomial is
            given by f*w1*w2*w3.
*/
            for (i1=0; i1<4; i1++) ll[i1]=l[i1+2];
            lloc=exll4s(ll,n);
            w=w1*w2*w3;
            p[lloc] += w*f;
         }
      }
   }
   return;
}
