/*
here we have included the routines that transform the hamiltonian, by
killing some preselected monomials.
*/

#include <stdio.h>
#include <stdlib.h>

#include "arit-c.h"

extern "C" {
#include "mp6s.h"
#include "io6s.h"
}

#include "basop6s.h"

void nf6s(complex **h, integer nh, integer n, char *nom,
          integer (*kill)(integer *))
/*
this is to put h (power expansion of degree nh) in normal form, up to
degree n. to control whether the generating function is written or not
to a file, we use the first character of the string nom (see below).

parameters:
h:    power expansion, handled by mp6s (input).
nh:   degree of h.
n:    degree of the normal form wanted. of course, n<=nh.
nom:  name of the (binary) file where the generation function of the
      transformation will be stored. if the name starts with a blank
      character (' '), this file is not written.
kill: pointer to a function that, given a monomial, returns 1 or 0
      when the monomial has to be killed or not.  */
{
   void cage(complex w[3], complex *p, complex *g, integer r,
             integer (*kill)(integer *));
   void traham(complex **h, integer n, complex *g, integer r,
             integer (*kill)(integer *), complex *u, complex *v);
   complex *g,*u,*v,w[3];
   integer j,k[6],r;
   char c='e';
   if (n > nh)
      {
         puts("nf6s error: n can't be greater than nh");
         printf("  n: %d\n nh: %d\n",n,nh);
         exit(1);
      }
/*
   getting the frequencies
*/
   k[0]=k[1]=1; k[2]=k[3]=k[4]=k[5]=0;
   j=exll6s(k,2);
   w[0]=h[2][j];
   k[2]=k[3]=1; k[0]=k[1]=k[4]=k[5]=0;
   j=exll6s(k,2);
   w[1]=h[2][j];
   k[4]=k[5]=1; k[0]=k[1]=k[2]=k[3]=0;
   j=exll6s(k,2);
   w[2]=h[2][j];
/*
   the generating function will be only computed up to degree n.
*/
   r=ntph6s(n);
   g=(complex*)malloc(r*sizeof(complex));
   if (g == 0) {puts("nf6s error: out of memory (1)"); exit(1);}
/*
   the working vectors will have to hold homogeneous polynomials up
   to degree nh.
*/
   r=ntph6s(nh);
   u=(complex*)malloc(r*sizeof(complex));
   if (u == 0) {puts("nf6s error: out of memory (2)"); exit(1);}
   v=(complex*)malloc(r*sizeof(complex));
   if (v == 0) {puts("nf6s error: out of memory (3)"); exit(1);}
/*
   main loop
*/
   for (r=3; r<=n; r++)
   {
      printf("nf6s: computing degree %d\n",r);
      cage(w,h[r],g,r,kill);
      if (nom[0] != ' ')
         {
            wpb6s(g,r,nom,c);
            c='a';
         }
      traham(h,nh,g,r,kill,u,v);
   }
   free(v);
   free(u);
   free(g);
   return;
}
void cage(complex w[3], complex *p, complex *g, integer r,
          integer (*kill)(integer *))
/*
this is to compute the generating function g that puts in normal
form the terms of degree r of the hamiltonian (these terms are
contained in p).

parameters:
w:    frequencies of the (linearized) hamiltonian at the equilibrium
      point (input).
p:    homogeneous polynomial to be put in normal form in the actual
      step (input).
g:    generating function that puts p in normal form (output).
r:    degree of p and g.
kill: pointer to a function that, given a monomial, returns 1 or 0
      when the monomial has to be killed or not.
*/
{
   complex f;
   integer j,k[6],m;
   m=ntph6s(r);
   for (j=0; j<m; j++)
   {
      llex6s(j,k,r);
/*
      next 'if' controls which monomials are going to be killed.
*/
      if ((*kill)(k))
         {
            f=w[0]*(k[1]-k[0])+w[1]*(k[3]-k[2])+w[2]*(k[5]-k[4]);
            g[j]=-(p[j]/f);
         }
         else
         {
            g[j]=0;
         }
   }
   return;
}
void traham(complex **h, integer n, complex *g, integer r,
            integer (*kill)(integer *), complex *u, complex *v)
/*
this is to transform the hamiltonian h by means of the generating
function g. the elements of h that we know must be zero (because of the
cancellations in the transformation) are explicitly zeroed.

parameter:
h: on input: hamiltonian to be transformed.
   on output: transformed hamiltonian.
n: degree of the expansion h.
g: generating function (an homogeneous polynomial of degree r) that
   puts the part of degree r of h in normal form (input).
r: degree of g.
u: used as internal working space. it must be big enough to contain an
   homogeneous polynomial of degree n.
v: same as u.
*/
{
   void put0(complex *p, integer r, integer (*kill)(integer *));
   renum f;
   integer i,j,k,l,m;
   printf("traham: starting degree %d...\n",r);
   for (i=n-r+2; i>1; i--)
   {
      printf("\n");
      m=ntph6s(i);
      for (k=0; k<m; k++) v[k]=h[i][k];
      f=1;
      l=0;
      for (j=i+r-2; j<=n; j+=r-2)
      {
         ++l;
         f*=l;
         printf("%3d,%3d -> %3d\n",j-r+2,r,j);
         papu6s(v,j-r+2,g,r,u,j,0);
         m=ntph6s(j);
         for (k=0; k<m; k++)
         {
            v[k]=u[k];
            h[j][k]+=u[k]/f;
         }
      }
   }
   put0(h[r],r,kill);
   printf("traham: degree %d ended\n",r);
   return;
}
void put0(complex *p, integer r, integer (*kill)(integer *))
/*
this is to put zeros in the places of p that correspond to coefficients
that have been killed by the normal form process. these coefficients are
not exactly zero because of the numerical errors. see the remarks in the
paper about this fact.

parameters:
p:    homogeneous polynomial of degree r. it corresponds to the
      monomials of degree r of the hamiltonian. on exit, the monomials
      killed by the normal form process have been explicitly zeroed
      (input/output).
r:    degree of p.
kill: pointer to a function that, given a monomial, returns 1 or 0
      when the monomial has to be killed or not.
*/
{
   integer j,k[6],m;
   m=ntph6s(r);
   for (j=0; j<m; j++)
   {
      llex6s(j,k,r);
/*
      next 'if' controls which monomials are going to be set to zero.
*/
      if ((*kill)(k)) p[j]=0;
   }
   return;
}
