/*
this program integrates numerically the restricted three body problem
(rtbp), with a fixed stepsize. basically, it reads a table of points
corresponding to the same orbit, and the time step between points is
the time step of integration. then, the program tests whether these
points belong to the same orbit of the rtbp or not.of course, the time
step used has to be small enough in order to ensure that the numerical
integration is accurate. the time steps used here (0.1) guarantees an
error smaller than 1.e-16 (of course, I'm skipping the roundoff
errors!), if one stays away from the two primaries. this is not a
problem here since the expansions we are using have a convergence
radius smaller than the distance to the closest primary. this program
is used to test the results obtained from the normal form and central
manifold computations.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "rk78.h"
#include "uneix.h"

static double mu; /* routine rtbp below must "see" this value */

/*
next lines are to ensure that the variable DATA is defined. it
must contain the directories where data files are going to be
stored/found. it is usually defined in the command line (see
makefile).
*/

#if !defined(DATA)
#define DATA ""
#endif

int main(void)
{
   void rtbp(double t, double *x, int n, double *y);
   double t,x[6],y[6],h,hmin,hmax,s;
   int j,k;
   char *nom;
   FILE *f1;
   nom=uneix(DATA,"rtbp.mu");
   f1=fopen(nom,"r");
   if (f1 == NULL) {printf("main-rtbp: cannot open %s\n",nom); exit(1);}
   fscanf(f1,"%le",&mu);
   fclose(f1);
   printf("mass parameter: %24.16e\n",mu);
   nom=uneix(DATA,"rtbp.dad");
   f1=fopen(nom,"r");
   if (f1 == NULL) {printf("main-rtbp: cannot open %s\n",nom); exit(1);}
   fscanf(f1,"%d %d",&j,&k);
   ini_rk78(6);
   k=0;
   while(fscanf(f1,"%le %le %le %le %le %le",x,x+1,x+2,x+3,x+4,x+5) != EOF)
   {
      if (k == 1)
         {
            for (j=0; j<6; j++) printf(" %e",fabs(y[j]-x[j]));
            printf("\n");
            s=0.e0;
            for (j=0; j<6; j++) s += (y[j]-x[j])*(y[j]-x[j]);
            s=sqrt(s);
            printf("%24.16e\n",s);
         }
      t=0.e0;
      h=0.1e0;
      hmin=0.1e0;
      hmax=0.1e0;
      rk78(&t,x,&h,1.e-14,hmin,hmax,6,rtbp);
      for (j=0; j<6; j++) y[j]=x[j];
      k=1;
   }
   fclose(f1);
   end_rk78(6);
   return 0;
}
void rtbp(double t, double *x, int n, double *y)
/*
this is the vectorfield of the restricted three body problem.

parameters:
t: adimensional time.
x: point in phase space (6 coordinates).
n: dimension of phase space (here it must be 6).
y: vectorfield at (t,x).
*/
{
   double r1,r13,r2,r23,aux3;
   r1=sqrt((x[0]-mu)*(x[0]-mu)+x[2]*x[2]+x[4]*x[4]);
   r13=r1*r1*r1;
   r2=sqrt((x[0]-mu+1)*(x[0]-mu+1)+x[2]*x[2]+x[4]*x[4]);
   r23=r2*r2*r2;
   aux3=(1-mu)/r13+mu/r23;
   y[0]=x[1]+x[2];
   y[1]=x[3]-(1-mu)*(x[0]-mu)/r13-mu*(x[0]-mu+1)/r23;
   y[2]=-x[0]+x[3];
   y[3]=-x[1]-aux3*x[2];
   y[4]=x[5];
   y[5]=-aux3*x[4];
   return;
}
