/*
this is to integrate (numerically) a trajectory in the central
manifold of l1, l2 or l3.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "uneix.h"
#include "rk78.h"

/*
as the hamiltonian is a quite sparse series (look into the file
cm.res), we are going to store it in a slightly different way (in
fact, this is not necessary at all, but I'm taking the opportunity of
showing an alternative way of handling polynomials that is very
suitable when only a few coefficients are present). we will use an
array to put the coefficients (without any particular ordering) and
four integer arrays to store the exponents of the corresponding
monomial. this is handled very easily using a structure. look at the
code to see how it works, and note that the hamiltonian is a global
variable. as we think the method is very natural, we do not make
further remarks.
*/

struct {
   double *c;    /* coef. of the hamiltonian */
   double *p[4]; /* working space */
   int **e;      /* exponents corresponding to the coef. of the ham. */
   int nc;       /* actual number of coefficients */
   int mc;       /* maxium number of coefficients */
   int g;        /* degree of the expansion */
} h;

/*
next lines are to ensure that the variable DATA is defined. it
must contain the directories where data files are going to be
stored/found. it is usually defined in the command line (see
makefile).
*/

#if !defined(DATA)
#define DATA ""
#endif

/* HARE contains the name of the file (in directory DATA) with the
expansion of the reduction to the central manifold.
*/

#define HARE "cm.res" /* hamiltonian reduced to the central manifold */

int main(void)
{
   void camp(double t, double *x, int n, double *f);
   void memo(int g);
   void llegeix(char *nom);
   double vham(double x[4]);
   double t,x[4],h,hmin,hmax;
   int j,g,np;
   char *hare,*ifi,*ofi;
   FILE *f;
   puts("degree to be used?");
   scanf("%d",&g);
   memo(g);
   hare=uneix(DATA,HARE);
   ifi=uneix(DATA,"nicm.dad");
   ofi=uneix(DATA,"nicm.res");
   llegeix(hare);
   f=fopen(ifi,"r");
   if (f == NULL) {printf("main: cannot open the file %s\n",ifi); exit(1);}
   fscanf(f,"%d",&np);
   fscanf(f,"%le %le %le %le",x,x+1,x+2,x+3);
   fclose(f);
   f=fopen(ofi,"w");
   if (f == NULL) {printf("cannot open the file %s\n",ofi); exit(1);}
   ini_rk78(4);
   t=0;
   h=0.1;
   hmin=0.1;
   hmax=0.1;
   printf("h=%25.16e %25.16e\n",h,vham(x));
   fprintf(f,"%d %d\n",np,4);
   fprintf(f,"%25.16e %25.16e %25.16e %25.16e\n",x[0],x[1],x[2],x[3]);
   for (j=1; j<np; j++)
   {
      rk78(&t,x,&h,1.e-14,hmin,hmax,4,camp);
      printf("%7d h=%25.16e %25.16e\n",j,h,vham(x));
      fprintf(f,"%25.16e %25.16e %25.16e %25.16e\n",x[0],x[1],x[2],x[3]);
   }
   fclose(f);
   end_rk78(4);
   return 0;
}
double vham(double x[4])
/*
this is to compute the value of the hamiltonian

parameters:
x: phase space point (input).

returned value: the value of the hamiltonian at x
*/
{
   double s;
   int i,j0,j1,j2,j3;
   h.p[0][0]=1; h.p[1][0]=1; h.p[2][0]=1; h.p[3][0]=1;
   for (i=1; i<=h.g; i++)
   {
      h.p[0][i]=x[0]*h.p[0][i-1];
      h.p[1][i]=x[1]*h.p[1][i-1];
      h.p[2][i]=x[2]*h.p[2][i-1];
      h.p[3][i]=x[3]*h.p[3][i-1];
   }
   s=0;
   for (i=0; i<h.nc; i++)
   {
      j0=h.e[i][0];
      j1=h.e[i][1];
      j2=h.e[i][2];
      j3=h.e[i][3];
      s+=h.c[i]*h.p[0][j0]*h.p[1][j1]*h.p[2][j2]*h.p[3][j3];
   }
   return(s);
}
void camp(double t, double *x, int n, double *f)
/*
this is the vectorfield related to the hamiltonian.

parameters:
t: time.
x: phase space point.
n: dimension of the phase space (here it must be 4).
f: value of the vectorfield at the point (t,x).
*/

{
   int i,j0,j1,j2,j3;
   h.p[0][0]=1; h.p[1][0]=1; h.p[2][0]=1; h.p[3][0]=1;
   for (i=1; i<=h.g; i++)
   {
      h.p[0][i]=x[0]*h.p[0][i-1];
      h.p[1][i]=x[1]*h.p[1][i-1];
      h.p[2][i]=x[2]*h.p[2][i-1];
      h.p[3][i]=x[3]*h.p[3][i-1];
   }
   f[0]=0; f[1]=0; f[2]=0; f[3]=0;
   for (i=0; i<h.nc; i++)
   {
      j0=h.e[i][0];
      j1=h.e[i][1];
      j2=h.e[i][2];
      j3=h.e[i][3];
      if (j1 != 0)
         f[0]+=j1*h.c[i]*h.p[0][j0]*h.p[1][j1-1]*h.p[2][j2]*h.p[3][j3];
      if (j0 != 0)
         f[1]-=j0*h.c[i]*h.p[0][j0-1]*h.p[1][j1]*h.p[2][j2]*h.p[3][j3];
      if (j3 != 0)
         f[2]+=j3*h.c[i]*h.p[0][j0]*h.p[1][j1]*h.p[2][j2]*h.p[3][j3-1];
      if (j2 != 0)
         f[3]-=j2*h.c[i]*h.p[0][j0]*h.p[1][j1]*h.p[2][j2-1]*h.p[3][j3];
   }
}
void memo(int g)
/*
this routine creates space to store the hamiltonian and related
things.

parameters:
g: maximum degree we want to store.
*/
{
   int i,j,m;
   h.g=g;
   m=0;
   for (i=2; i<=g; i++)
   {
      for (j=i; j>=0; j-=2) m += (j+1)*(i-j+1);
   }
   h.mc=m;
   h.c=(double*)malloc(m*sizeof(double));
   if (h.c == NULL) {printf("memo: out of memory (1). %d\n",m); exit(1);}
   h.e=(int**)malloc(m*sizeof(int*));
   if (h.e == NULL) {printf("memo: out of memory (2). %d\n",m); exit(1);}
   for (i=0; i<m; i++)
   {
      h.e[i]=(int*)malloc(4*sizeof(int));
      if (h.e[i] == NULL) {printf("memo: out of memory (3). %d\n",i); exit(1);}
   }
   for (i=0; i<4; i++)
   {
      h.p[i]=(double*)malloc((g+1)*sizeof(double));
      if (h.p[i] == NULL) {printf("memo: out of memory (4).\n"); exit(1);}
   }
   h.nc=0;
}
void llegeix(char *nom)
/*
this routine reads the reduced hamiltonian.

parameters:
nom: string with the name of the reduced hamiltonian
*/
{
   double x;
   int k,i0,i1,i2,i3,g;
   FILE *f;
   f=fopen(nom,"r");
   if (f == NULL) {printf("cannot open the file %s.\n",nom); exit(1);}
   k=0;
   while (fscanf(f,"%d %d %d %d %le",&i0,&i1,&i2,&i3,&x) != EOF)
   {
      if (fabs(x) < 1.e-12) continue;
      g=i0+i1+i2+i3;
      if (g > h.g) break;
      if (k == h.mc) {printf("llegeix error: %d\n",k); exit(1);}
      h.c[k]=x;
      h.e[k][0]=i0;
      h.e[k][1]=i1;
      h.e[k][2]=i2;
      h.e[k][3]=i3;
      ++k;
   }
   printf("total of read monomials: %d\n",k);
   h.nc=k;
   fclose(f);
}
