/*
this file contains the routines that initialize the realification.
these routines are independent of the kind of realification we are
going to do. they simply compute the realifications of the different
powers of a couple of position-momentum variables (that is, a
two-variables expansion). then, to realify a monomial with more
variables, the realification is done monomial by monomial, considering
each one as a product of the realifications of different couples
position-momentum.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

#include "arit-c.h"

extern "C" {
#include "mp3.h"
}

complex **ini_real(integer n)
/*
this is to allocate, fill up and return (a pointer to) a table used
during the realification. this table contains the realifications of
the monomials of two variables (position and momentum) up to degree n.
note that this routine calls imp3, and routine end_real calls amp3.

parameters:
n: degree of the expansion to be realified.

returned value: it is a table stored as a three-variables power
   expansion of degree 2*n. to explain the contents of this table, let
   us call q=q(x,y) and p=p(x,y) the realifying change of variables
   for a single variable (we stress that it is a linear change), where
   (q,p) are the complex coordinates and (x,y) the real ones. note
   that this change is the same for all the (complexified) variables
   (except for, of course, the subindices of the variables). then, the
   coefficient of the monomial (m1,m2,m3) in the table is the
   coefficient of (x^(m1+m2-m3))*(y^m3) in the polinomial
   (q^m1)*(p^m2).
*/
{
   complex coef(integer b, integer k, integer j, double **cb, renum *pr);
   integer i,j,k,l,m[3],lloc,n2,ng;
   double **cb;
   complex wq,wp,**cm2;
   renum *pr,r2;
   n2=2*n;
/*
   mp3 is initialized. the value returned is the amount (in Kb.) of
   memory allocated. We don't print this value.
*/
   i=imp3(n2);
/*
   we allocate memory for the table cm2 (we use calloc to fill it with
   zeros)
*/
   cm2=(complex**)malloc((n2+1)*sizeof(complex*));
   if (cm2 == NULL) {puts("ini_real: out of memory (1)."); exit(1);}
   for (j=0; j<=n2; j++)
   {
      k=ntph3(j);
      cm2[j]=(complex*)calloc(k,sizeof(complex));
      if (cm2[j] == NULL) {puts("ini_real: out of memory (2)."); exit(1);}
   }
/*
   in the next lines we create a working table with the binomial
   coefficients (cb[i][j] will contain i over j)
*/
   cb=(double**)malloc((n+1)*sizeof(double*));
   if (cb == NULL) {puts("ini_real: out of memory (3)"); exit(1);}
   for (i=0; i<=n; i++)
   {
      cb[i]=(double*)malloc((i+1)*sizeof(double));
      if (cb[i] == NULL) {puts("ini_real: out of memory (4)"); exit(1);}
   }
   cb[0][0]=1; cb[1][0]=1; cb[1][1]=1;
   for (i=2; i<=n; i++)
   {
      cb[i][0]=1;
      for (j=1; j<i; j++) cb[i][j]=cb[i-1][j-1]+cb[i-1][j];
      cb[i][i]=1;
   }
/*
   pr is an array to store negative powers of sqrt(2) (pr[j] will
   contain (sqrt(2))^(-j))
*/
   pr=(renum*)malloc((n+1)*sizeof(renum));
   if (pr == NULL) {puts("ini_real error: no memo (5)"); exit(1);}
   r2=1.e0/sqrt(2.e0);
   pr[0]=1.e0;
   for (i=1; i<=n; i++) pr[i]=pr[i-1]*r2;
/*
   the next block is the computation of the table cm2. variable i
   denotes the actual degree, k refers to the actual monomial in the
   variables (q,p) (in fact, this monomial is q^(i-k)p^k), j refers
   to the corresponding monomial in the (x,y) variables inside q^(i-k)
   (this is a monomial like x^(i-k-j)y^j) and finally, l refers to the
   monomials in the (x,y) variables inside p^k (this is a monomial
   like x^(k-l)y^l).
*/
   for (i=0; i<=n; i++)
   {
      for (k=0; k<=i; k++)
      {
         m[0]=i-k;
         m[1]=k;
         for (j=0; j<=m[0]; j++)
         {
            wq=coef(1,m[0],j,cb,pr);
            for (l=0; l<=m[1];l++)
            {
               wp=coef(2,m[1],l,cb,pr);
               m[2]=j+l;
               ng=i+m[2];
               lloc=exll3(m,ng);
/*
               note that in the next line we put += instead of = because
               there is more than one monomial for each exponent
*/
               cm2[ng][lloc] += wq*wp;
            }
         }
      }
   }
/*
   finally we free the working memory
*/
   free(pr);
   for (i=0; i<=n; i++) free(cb[i]);
   free(cb);
   return(cm2);
}
void end_real(complex **cm2, integer n)
/*
this is to free the memory used by the table cm2 during the
realification, and to free the internal tables of mp3.

parameters:
cm2: pointer to a table used during realification. this is the pointer
     returned by routine ini_real (input).
n:   degree of the realified expansion. it must coincide with the 'n'
     given to routine ini_real.
*/
{
   integer j,n2;
   n2=2*n;
   for (j=0; j<=n2; j++) free(cm2[j]);
   free(cm2);
   amp3();
   return;
}
complex coef(integer b, integer k, integer j, double **cb, renum *pr)
/*
this routine computes the coefficient of the monomial x^(k-j)y^j
in q^k or p^k, depending on the value of the flag b.

parameters:
b:  flag. if it is 1, the computation is done for q^k, otherwise it
    is done for p^k.
k:  exponent of q or p.
j:  number of monomial in the variables (x,y) in q^k or p^k (the
    first one is x^k and the last one is y^k).
cb: table with the binomial coefficients (input).
pr: table with negative powers of sqrt(2) (input).

returned value: the coefficient of the monomial x^(k-j)y^j.
*/
{
   complex w,z;
   integer l;
   if (b == 1) {l=j%4;} else {l=(k-j)%4;}
/*
   next lines are to compute (-sqrt(-1))^l
*/
   switch (l)
   {
      case 0:
         w=complex( 1.e0, 0.e0);
         break;
      case 1:
         w=complex( 0.e0,-1.e0);
         break;
      case 2:
         w=complex(-1.e0, 0.e0);
         break;
      case 3:
         w=complex( 0.e0, 1.e0);
         break;
      default:
         puts("coef error: l must be 0,1,2 or 3.");
         exit(1);
         break;
   }
/*
   hence, the coefficient is
*/
   z=(pr[k]*cb[k][j])*w;
   return(z);
}
