/*
this file contains the routines used to produce a power expansion of
the hamiltonian of the 3-d restricted three body problem around L5.
it is assumed that the mass ratio is below the Routh critical value,
this is, that L5 is linearly stable.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

extern "C" {
#include "mp6s.h"
}

#include "arit-c.h"
#include "basop6s.h"

#define G1 4  /*number of monomials of degree 1 (to allocate working space)*/
#define G2 13 /*number of monomials of degree 2 (to allocate working space)*/

void ccvl5(renum mu, complex c[6][6], renum m[6][6])
/*
this routine computes the (complex, symplectic) change of
variables that diagonalizes the linearized vectorfield around L5.
the order of the variables is (x,px,y,py,z,pz).

parameters:
mu: mass ratio, 0 < mu < mu_R=0.03852 (input).
c:  complex change of variables. (output).
m:  real change of variables. it only involves the planar variables since
    in the vertical direction the change is the identity (output).
*/
{
   renum a,d,w[2],r,u,v,wj2,wj4,tmp,zero;
   integer i,j;
   if (mu > 0.03852) {puts("ccvl5 error: mu is bigger than Routh"); exit(1);}
   for (i=0; i<6; i++)
   {
      for (j=0; j<6; j++) c[i][j]=0;
      for (j=0; j<6; j++) m[i][j]=0;
   }
   tmp=27;
   a=-(1-2*mu)*sqrt(tmp)/4;
   d=tmp/16-a*a;
   d=1-4*d;
   d=sqrt(d);
   w[0]=sqrt((1+d)/2);
   w[1]=-sqrt((1-d)/2);
   for (j=0; j<2; j++)
   {
/*
      first we compute the complex change
*/
      i=2*j;
      wj2=w[j]*w[j];
      wj4=wj2*wj2;
      r=sqrt(w[j]*(4*wj4+wj2-1.5));
      u=a/r;
      v=2*w[j]/r;
      c[0][i]=complex(u,v);
      c[0][i+1]=complex(v,u);
      u=(0.75-wj2)/r;
      v=a*w[j]/r;
      c[1][i]=complex(u,v);
      c[1][i+1]=complex(v,u);
      u=(-0.75-wj2)/r;
      v=0;
      c[2][i]=complex(u,v);
      c[2][i+1]=complex(v,u);
      u=a/r;
      v=(1.25-wj2)*w[j]/r;
      c[3][i]=complex(u,v);
      c[3][i+1]=complex(v,u);
/*
      and then the real one
*/
      r=sqrt(w[j]*(2*wj4+0.5*wj2-0.75));
      m[0][i]=a/r;
      m[0][i+1]=2*w[j]/r;
      m[1][i]=(0.75-wj2)/r;
      m[1][i+1]=a*w[j]/r;
      m[2][i]=(-0.75-wj2)/r;
      m[2][i+1]=0;
      m[3][i]=a/r;
      m[3][i+1]=(1.25-wj2)*w[j]/r;
   }
   zero=0;
   tmp=2;
   r=1/sqrt(tmp);
   c[4][4]=complex(r,zero);
   c[5][5]=complex(r,zero);
   c[5][4]=complex(zero,r);
   c[4][5]=complex(zero,r);
   m[4][4]=1;
   m[5][5]=1;
   return;
}
void exp_l5(renum mu, complex c[6][6], integer n, complex **h)
/*
this is to expand the hamiltonian. the expansion is complexified in
order to put h2 in normal form.

parameters:
mu: mass ratio.
c:  change of variables computed by ccvl5 (input).
n:  degree of the desired expansion.
h:  expanded hamiltonian (output).
*/
{
   void exrec(renum xl, renum yl, complex *x, complex *y,
          complex **h, integer nor, complex *q, integer bn);
   void reste(renum alf, renum bet, complex *x, complex *px,
          complex *y, complex *py, complex **h, integer nor, complex *q);
   complex px[G1],py[G1],x[G1],y[G1],q[G2],**w;
   renum xs,ys,xj,yj,alf,bet,umu,tmp,zero;
   integer k[6],j,m,r;

   zero=0;
/*
   this is to initialize the equilibrium point (l5).
*/
   xs=0.5;
   tmp=3;
   ys=-sqrt(tmp)/2;
   xj=-xs;
   yj=ys;
   alf=mu-0.5;
   bet=sqrt(tmp)/2;
/*
   loading the linear change of variables, to get the expansion
   in linear normal form
*/
   for (j=0; j<4; j++)
   {
      x[j]=c[0][j];
      px[j]=c[1][j];
      y[j]=c[2][j];
      py[j]=c[3][j];
   }
/*
   we need working space (of the same type and size as h)
*/
   w=(complex**)malloc((n+1)*sizeof(complex*));
   if (w == 0) {puts("exp_l5: out of memory (1)"); exit(1);}
   for (j=0; j<=n; j++)
   {
      m=ntph6s(j);
      w[j]=(complex*)malloc(m*sizeof(complex));
      if (w[j] == 0) {puts("exp_l5: out of memory (2)"); exit(1);}
   }
/*
   now we apply the recurrences
*/
   puts("exp_l5: starting first recurrence...");
   exrec(xs,ys,x,y,w,n,q,1);
   puts("exp_l5: starting second recurrence...");
   exrec(xj,yj,x,y,h,n,q,0);
   puts("exp_l5: recurrences ended.");
   umu=1-mu;
   for (r=0; r<=n; r++)
   {
      m=ntph6s(r);
      for (j=0; j<m; j++) h[r][j]=mu*h[r][j]+umu*w[r][j];
   }
   for (j=0; j<=n; j++) free(w[j]);
   free(w);
/*
   the remaining terms of the hamiltonian are added
*/
   reste(alf,bet,x,px,y,py,h,n,q);
/*
   this is to put 0 where we know it should be zero (it is only for
   aesthetical reasons: when printing the hamiltonian in ascii format,
   it is nicer to get 0 rather than 0.xxxxxxxxx...e-15)
*/
   m=ntph6s(2);
   for (j=0; j<m; j++)
   {
      llex6s(j,k,2);
      h[2][j]=complex(zero,imag(h[2][j]));
      if ((k[0] == 1) && (k[1] == 1)) continue;
      if ((k[2] == 1) && (k[3] == 1)) continue;
      if ((k[4] == 1) && (k[5] == 1)) continue;
      h[2][j]=0;
   }
   puts("exp_l5: expansion ended.");
   return;
}
void exrec(renum xl, renum yl, complex *x, complex *y, complex **h,
           integer nor, complex *q, integer bn)
/*
this performs the expansion of one of the nonlinear terms of the
hamiltonian, using a recurrence based on the one of legendre's
polynomials.

parameters:
xl,yl: coordinates of one of the masses.
x,y:   vectors containing the change of variables (for the x and y
       variables) that diagonalizes h2 (input).
h:     final expansion (output).
nor:   degree of the expansion wanted.
q:     it contains x*x+y*y+z*z (output).
bn:    flag. if it is equal to 1 indicates that q already contains
       the value x*x+y*y+z*z, otherwise q must be initialized.
*/
{
   complex r[G1],rr[G1],qq[G2];
   renum w1,w2,tmp;
   integer n,i,j,k[6];
   for (j=0; j<G1; j++) r[j]=xl*x[j]+yl*y[j];
   if (bn == 1)
      {
         pph6s(x,1,x,1,q,2,0);
         pph6s(y,1,y,1,q,2,1);
         for (j=0; j<6; j++) k[j]=0;
         k[4]=2;
         i=exll6s(k,2);
         q[i] += 0.5;
         k[4]=0;
         k[5]=2;
         i=exll6s(k,2);
         q[i] -= 0.5;
         k[4]=1;
         k[5]=1;
         i=exll6s(k,2);
         q[i] += complex(0,1);
      }
/*
   next lines are the recurrence
*/
   h[0][0]=complex(1,0);
   for (j=0; j<G1; j++) h[1][j]=r[j];
   for (n=1; n<nor; n++)
   {
      printf("computing degree %d\n",n+1);
      tmp=n+1;
      w1=(2*n+1)/tmp;
      for (j=0; j<G1; j++) rr[j]=w1*r[j];
      w2=-n/tmp;
      for (j=0; j<G2; j++) qq[j]=w2*q[j];
      pph6s(h[n],n,rr,1,h[n+1],n+1,0);
      pph6s(h[n-1],n-1,qq,2,h[n+1],n+1,1);
   }
   return;
}
void reste(renum alf, renum bet, complex *x, complex *px,
       complex *y, complex *py, complex **h, integer nor, complex *q)
/*
this routine computes the terms of the expansion that are not
obtained from the recurrences.

parametres:
alf,bet:   coordinates of the equilibrium point.
x,px,y,py: change of coordinates (input).
h:         hamiltonian (input/output).
nor:       degree of the hamiltonian.
q:         working space.
*/
{
   integer i,j,k[6],m;
   for (j=0; j<G1; j++) h[1][j] += alf*x[j]+bet*y[j];
   for (i=0; i<=nor; i++)
   {
      m=ntph6s(i);
      for (j=0; j<m; j++) h[i][j]=-h[i][j];
   }
   pph6s(px,1,px,1,q,2,0);
   pph6s(py,1,py,1,q,2,1);
   for (j=0; j<6; j++) k[j]=0;
   k[4]=2;
   i=exll6s(k,2);
   q[i] -= 0.5;
   k[4]=0;
   k[5]=2;
   i=exll6s(k,2);
   q[i] += 0.5;
   k[4]=1;
   k[5]=1;
   i=exll6s(k,2);
   q[i] += complex(0,1);
   for (j=0; j<G2; j++) h[2][j] += 0.5*q[j];
   pph6s(x,1,py,1,q,2,0);
   for (j=0; j<G2; j++) q[j]=-q[j];
   pph6s(y,1,px,1,q,2,1);
   for (j=0; j<G2; j++) h[2][j] += q[j];
   return;
}
