/*
this file contains the routines used to produce a power expansion of
the hamiltonian of the spatial restricted three body problem around
one of the collinear points L1, L2 or L3.
*/

#include <math.h>
#include <stdio.h>
#include <stdlib.h>

extern "C" {
#include "mp6s.h"
}

#include "arit-c.h"
#include "basop6s.h"

#define G1 4  /*number of monomials of degree 1 (to allocate working space)*/
#define G2 13 /*number of monomials of degree 2 (to allocate working space)*/

renum ccvl123(renum mu, complex c[6][6], renum m[6][6], integer li)
/*
this routine computes the (complex, symplectic) change of variables
that diagonalizes the linearized vectorfield around a collinear point.
the order of the variables is (x,px,y,py,z,pz).

parameters:
mu: mass ratio, 0 < mu <= 0.5 (input).
c:  complex change of variables (output).
m:  real change of variables (output).
li: it must be 1, 2 or 3, according to the equilibrium point we are
    interested in.
*/
{
   renum solv_euler(renum mu, integer l);
   renum cn(integer n, integer l, renum mu, renum gam);
   renum check_sym(complex c[6][6]);
   complex cc[6][6];
   renum gam,c2,r,la,dla,w1,dw1,w2;
   integer i,j,k;
   if ((li < 1) || (li > 3)) {puts("ccvl123: li must be 1, 2 or 3"); exit(1);}
   for (i=0; i<6; i++)
   {
      for (j=0; j<6; j++) {m[i][j]=0; cc[i][j]=0;}
   }
   gam=solv_euler(mu,li);
   c2=cn(2,li,mu,gam);
/*
   now we compute the eigenvalues of the linearized vectorfield
*/
   r=sqrt(c2*(9*c2-8));
   la=sqrt((c2-2+r)/2);
   dla=sqrt(2*la*((4+3*c2)*la*la+4+5*c2-6*c2*c2));
   w1=sqrt(-(c2-2-r)/2);
   dw1=sqrt(w1*((4+3*c2)*w1*w1-4-5*c2+6*c2*c2));
   w2=sqrt(c2);
/*
   next, the real change of variable is obtained
*/
   m[0][0]=2*la;                    m[0][0]/=dla;
   m[1][0]=la*la+2*c2+1;            m[1][0]/=dla;
   m[2][0]=la*la-2*c2-1;            m[2][0]/=dla;
   m[3][0]=la*(la*la+(1-2*c2));     m[3][0]/=dla;
   m[0][1]=-2*la;                   m[0][1]/=dla;
   m[1][1]=la*la+2*c2+1;            m[1][1]/=dla;
   m[2][1]=la*la-2*c2-1;            m[2][1]/=dla;
   m[3][1]=-la*(la*la+(1-2*c2));    m[3][1]/=dla;
   m[1][2]=-w1*w1+2*c2+1;           m[1][2]/=dw1;
   m[2][2]=-w1*w1-2*c2-1;           m[2][2]/=dw1;
   m[0][3]=2*w1;                    m[0][3]/=dw1;
   m[3][3]=w1*(-w1*w1+(1-2*c2));    m[3][3]/=dw1;
   m[4][4]=1/sqrt(w2);
   m[5][5]=sqrt(w2);
/*
   this is the complexifying change
*/
   r=2;
   r=1/sqrt(r);
/*
   the first two variables correspond to the hyperbolic directions, so
   they are not complexified
*/
   cc[0][0]=1;
   cc[1][1]=1;
   for (j=2; j<6; j+=2)
   {
      cc[j][j]=complex(r,0);
      cc[j][j+1]=complex(0,r);
      cc[j+1][j]=complex(0,r);
      cc[j+1][j+1]=complex(r,0);
   }
/*
   we compute the final change
*/
   for (i=0; i<6; i++)
   {
      for (j=0; j<6; j++)
      {
         c[i][j]=m[i][0]*cc[0][j];
         for (k=1; k<6; k++) c[i][j] += m[i][k]*cc[k][j];
      }
   }
/*
   next lines are optional. they check that the final change
   is symplectic.
*/
   r=check_sym(c);
   if (r > 1.e-14) printf("WARNING: error of symplect.: %24.16e\n",r);
   return(gam);
}
renum cn(integer n, integer l, renum mu, renum gam)
/*
this function returns the coefficient c_n that appears in the
expansion of the nonlinear terms of the hamiltonian.

parameters:
n:   we look for the coefficient of the terms of order n
l:   number of the point (1, 2 or 3).
gam: the solution of the corresponding euler equation.

returned value: the coefficient we are asking for.
*/
{
   renum c,u,v;
   integer s;
   if (n < 2) {printf("cn error (n<2). n=%d\n",n); exit(1);}
   u=1/(gam*gam*gam);
   s=((n%2) == 0) ? 1 : -1;
   switch (l)
   {
      case 1:
         v=pow(gam/(1-gam),n+1);
         c=u*(mu+s*(1-mu)*v);
         break;
      case 2:
         v=pow(gam/(1+gam),n+1);
         c=u*s*(mu+(1-mu)*v);
         break;
      case 3:
         v=pow(gam/(1+gam),n+1);
         c=s*u*(1-mu+mu*v);
         break;
      default:
         printf("cn error: l must be 1, 2 or 3. l=%d\n",l);
         puts("action taken: program aborted");
         exit(1);
   }
   return(c);
}
renum solv_euler(renum mu, integer l)
/*
this is to compute the solution of the euler equation, by means of
the newton method.

parameters:
mu: mass ratio (0 < mu <= 0.5).
l:  point. it must  be 1, 2 or 3.

returned value: the only positive solution of the euler's quintic equation
*/
{
   renum euler(renum ro, renum mu, integer j);
   renum d_euler(renum ro, renum mu, integer j);
   renum ro,f,df,h,err;
   integer k;
   err=1.e-15;
/*
   we first compute a initial guess
*/
   ro=((l == 1) || (l == 2)) ? exp(log(mu/3)/3) : 1-(7.e0/12.e0)*mu;
   k=0;
/*
   next loop performs the newton iterations
*/
   do {
      ++k;
      if (k == 10)
         {
            puts("solv_euler error. newton doesn't converge!");
            puts("action taken: program aborted");
            exit(1);
         }
      f=euler(ro,mu,l);
      df=d_euler(ro,mu,l);
      h=f/df;
      ro -= h;
   } while(fabs(f) > err);
   return(ro);
}
renum euler(renum ro, renum mu, integer j)
/*
this is to evaluate the euler quintic polynomial.

parameters:
ro: independent variable.
mu: mass parameter.
j:  number (1, 2 or 3) of the collinear point.

returned value: value of the quintic polynomial.
*/
{
   renum p;
   switch (j) {
      case 1:
         p=((((ro-3+mu)*ro+3-2*mu)*ro-mu)*ro+2*mu)*ro-mu;
         break;
      case 2:
         p=((((ro+3-mu)*ro+3-2*mu)*ro-mu)*ro-2*mu)*ro-mu;
         break;
      case 3:
         p=((((ro+2+mu)*ro+1+2*mu)*ro-1+mu)*ro-2+2*mu)*ro-1+mu;
         break;
      default:
         printf("euler error. j=%d\n",j);
         puts("action taken: program aborted"),
         exit(1);
   }
   return(p);
}
renum d_euler(renum ro, renum mu, integer j)
/*
this is to evaluate the derivative of the euler quintic polynomial.

parameters:
ro: independent variable.
mu: mass parameter.
j:  number (1, 2 or 3) of the collinear point.

returned value: value of the derivative of the quintic polynomial.
*/
{
   renum p;
   switch (j) {
      case 1:
         p=(((5*ro-4*(3-mu))*ro+3*(3-2*mu))*ro-2*mu)*ro+2*mu;
         break;
      case 2:
         p=(((5*ro+4*(3-mu))*ro+3*(3-2*mu))*ro-2*mu)*ro-2*mu;
         break;
      case 3:
         p=(((5*ro+8+4*mu)*ro+3+6*mu)*ro-2+2*mu)*ro-2+2*mu;
         break;
      default:
         printf("d_euler error. j=%d\n",j);
         puts("action taken: program aborted"),
         exit(1);
   }
   return(p);
}
renum check_sym(complex c[6][6])
/*
this is to check whether the matrix c is symplectic or not.

parameters:
c: matrix to be tested (input).

returned value: the biggest entry (in absolute value) of the matrix
   transpose(c)*j*c-j.
*/
{
   complex ct[6][6],a[6][6],b[6][6];
   renum j[6][6],e,f,u,v;
   integer i,k,l;
   for (i=0; i<6; i++) {for (k=0; k<6; k++) {j[i][k]=0; ct[i][k]=c[k][i];}}
   j[0][1]=1; j[1][0]=-1;
   j[2][3]=1; j[3][2]=-1;
   j[4][5]=1; j[5][4]=-1;
   for (i=0; i<6; i++)
   {
      for (k=0; k<6; k++)
      {
         a[i][k]=ct[i][0]*j[0][k];
         for (l=1; l<6; l++) a[i][k] += ct[i][l]*j[l][k];
      }
   }
   for (i=0; i<6; i++)
   {
      for (k=0; k<6; k++)
      {
         b[i][k]=a[i][0]*c[0][k];
         for (l=1; l<6; l++) b[i][k] += a[i][l]*c[l][k];
      }
   }
   e=0;
   for (i=0; i<6; i++)
   {
      for (k=0; k<6; k++)
      {
         u=real(b[i][k])-j[i][k];
         v=imag(b[i][k]);
         f=sqrt(u*u+v*v);
         e=(f>e) ? f : e;
      }
   }
   return(e);
}
void exp_l123(renum mu, complex c[6][6], integer n, complex **h, integer li)
/*
this is to expand the hamiltonian around L1, L2 or L3. the expansion
is complexified such that h2 is in normal form.

parameters:
mu: mass ratio.
c:  change of variables computed by ccvl123 (input).
n:  degree of the desired expansion.
h:  expanded hamiltonian (output).
li: collinear point. it must be 1, 2 or 3.
*/
{
   void exrecl123(complex *x, complex *y, complex *z, integer nor, renum g,
              renum mu, integer l, complex **h, complex *q);
   void afz2(complex *z, complex *q);
   renum solv_euler(renum mu, integer l);
   complex px[G1],py[G1],x[G1],y[G1],q[G2],z[2],pz[2];
   renum g;
   integer k[6],i,j,l,m;
   if (G1 != 4) {puts("exp_l123 error: wrong constant"); exit(1);}
   if ((li != 1) && (li != 2) && (li != 3))
      {
         printf("exp_l123 error: li is %d and it must be 1, 2 or 3\n",li);
         exit(1);
      }
/*
   we copy the change of variables from the matrix to vectors
   corresponding to linear polynomials. in this way the change will be
   handled by mp6s
*/
   for (i=0; i<4; i++)
   {
      for (j=0; j<6; j++) k[j]=0;
      k[i]=1;
      l=exll6s(k,1);
      x[i]=c[0][l];
      px[i]=c[1][l];
      y[i]=c[2][l];
      py[i]=c[3][l];
   }
/*
   the change for the variables (z,pz) is stored in a different way.
   as this change is uncoupled from the planar directions, we only
   store the two components of the vertical direction. Moreover,
   this change does not conform to the symmetry mp6s. As this term
   appears squared in the hamiltonian, we will compute this square
   from these two components and we will store it in as usual, since
   the square conforms to the mp6s symmetry.
*/
   z[0]=c[4][4];
   z[1]=c[4][5];
   pz[0]=c[5][4];
   pz[1]=c[5][5];
/*
   now we apply the recurrence
*/
   g=solv_euler(mu,li);
   puts("exp_l123: starting recurrence...");
   exrecl123(x,y,z,n,g,mu,li,h,q);
   puts("exp_l123: recurrence ended.");
/*
   the remaining terms of the hamiltonian are added
*/
   pph6s(px,1,px,1,q,2,0);
   pph6s(py,1,py,1,q,2,1);
   afz2(pz,q);
   for (j=0; j<G2; j++) h[2][j] += 0.5*q[j];
   pph6s(y,1,px,1,q,2,0);
   for (j=0; j<G2; j++) h[2][j] += q[j];
   pph6s(x,1,py,1,q,2,0);
   for (j=0; j<G2; j++) h[2][j] -= q[j];
/*
   this is to put 0 where we know it should be zero.
   here we use that the first two variables correspond to
   the hyperbolic direction.
*/
   m=ntph6s(2);
   for (j=0; j<m; j++)
   {
      llex6s(j,k,2);
      if ((k[0] == 1) && (k[1] == 1))
         {
            h[2][j]=complex(real(h[2][j]),0);
            continue;
         }
      if (((k[2] == 1) && (k[3] == 1)) || ((k[4] == 1) && (k[5] == 1)))
         {
            h[2][j]=complex(0,imag(h[2][j]));
            continue;
         }
      h[2][j]=0;
   }
   puts("exp_l123: expansion ended.");
   return;
}
void exrecl123(complex *x, complex *y, complex *z, integer nor, renum g,
           renum mu, integer l, complex **h, complex *q)
/*

this performs the expansion of the nonlinear part of the hamiltonian
for the case L1, L2 or L3, using a recurrence based on the one of
legendre polynomials.

parameters:
x,y,z: vector containing the change of variables (for the x, y and
       z variables) that diagonalizes h2 (input).
q:     working space (big enough to store x*x+y*y+z*z).
nor:   degree of the expansion wanted.
g:     solution of euler's quintic for this case.
mu:    mass parameter.
l:     collinear point (it must be 1, 2 or 3).
h:     space to store the final expansion (output).
*/
{
   renum cn(integer n, integer l, renum mu, renum gam);
   void afz2(complex *z, complex *q);
   complex rr[G1],qq[G2];
   renum a,b,c,un;
   integer n,j,m;
   un=1;
   for (n=0; n<=nor; n++)
   {
      m=ntph6s(n);
      for (j=0; j<m; j++) h[n][j]=0;
   }
/*
   next line is to facilitate the implementation of the recurrence.
   the terms of order 1 of h will be zeroed at the end of this
   computation.
*/
   for (j=0; j<G1; j++) h[1][j]=x[j];
/*
   to simplify the second degree terms of the recurrence, we
   re-write them in the form x^2-(1/2)*(y^2+z^2)
*/
   pph6s(y,1,y,1,q,2,0);
   afz2(z,q);
   for (j=0; j<G2; j++) h[2][j]=-0.5*q[j];
   pph6s(x,1,x,1,h[2],2,1);
   pph6s(x,1,x,1,q,2,1);
/*
   next lines are the recurrence
*/
   for (n=3; n<=nor; n++)
   {
      printf("exrecl123: doing degree %d\n",n);
      a=2-un/n;
      b=-1+un/n;
      for (j=0; j<G1; j++) rr[j]=a*x[j];
      for (j=0; j<G2; j++) qq[j]=b*q[j];
      pph6s(h[n-2],n-2,qq,2,h[n],n,0);
      pph6s(h[n-1],n-1,rr,1,h[n],n,1);
   }
/*
   to finish, let us take into account the constants c_n. As this
   expansion appears with a minus sign in the expression of the
   hamiltonian, we will take the opportunity of changing its sign by
   adding a minus in front of the constants c_n.
*/
   for (n=2; n<=nor; n++)
   {
      c=-cn(n,l,mu,g);
      m=ntph6s(n);
      for (j=0; j<m; j++) h[n][j] *= c;
   }
   for (j=0; j<G1; j++) h[1][j]=0;
   return;
}
void afz2(complex *z, complex *q)
/*
this is to compute the polynomial (z[0]*z+z[1]*pz)^2 (here, z and pz
denote the variables), and to add the result to the second degree
polynomial contained in q.

parameters:
z: vector with two coeffcients (input).
q: where the result is added (input/output).
*/
{
   integer k[6],j;
   k[4]=2;
   k[0]=k[1]=k[2]=k[3]=k[5]=0;
   j=exll6s(k,2);
   q[j] += z[0]*z[0];
   k[5]=2;
   k[0]=k[1]=k[2]=k[3]=k[4]=0;
   j=exll6s(k,2);
   q[j] += z[1]*z[1];
   k[4]=1;
   k[5]=1;
   k[0]=k[1]=k[2]=k[3]=0;
   j=exll6s(k,2);
   q[j] += 2*z[0]*z[1];
   return;
}
