/*
this file contains the routines that compute the power expansiond for
the changes of variables. as the symmetries are different according to
the variable considered (the change for the first 4 variables is of
the kind mp6s and the change for the last 2 is of the kind mp6p), we
have included routines for all these cases. we want to stress that the
routines in this file only compute the (nonlinear) change
corresponding to the normal form computation, that is, the change that
goes from the initial (and complex) hamiltonian with h2 in diagonal
form to the final (and still complex) normal form. for completeness,
we have also included the computation of the inverse change. we recall
that the linear change that puts h2 in the suitable form is not
computed in this file.
*/

#include <stdio.h>
#include <stdlib.h>

#include "arit-c.h"

extern "C" {
#include "mp6s.h"
#include "mp6p.h"
#include "io6s.h"
}

#include "basop6s.h"
#include "basop6sp.h"

void ccvd6s(integer vn, complex **c, integer n, char *nom)
/*
routine to compute the (complex) change of variables corresponding
to variable number vn. it is assumed that the change is handled by
mp6s (this is true for vn=1,2,3 and 4 in the RTBP case).

parameters:
vn:  number (between 1 and 4) pointing to the variable for which the
     transformation should be computed. the numbering of the variables
     is the same for all this work: 1 corresponds to the first position,
     2 to the first momentum, 3 to the second position and 4 to the
     second momentum.
c:   space to store the computed change of variables. this space must
     be big enough to store expansions --with the simmetry mp6s-- of
     degree n (output).
n:   degree up to which the change must be computed.
nom: name of the (binary) file containing the generating function of
     the change of variables. it is assumed that the generating function
     starts from degree 3 up to degree at least n. this file must have
     been written by the routines of io6s.c (input).
*/
{
   complex *g,*u,*v;
   renum f;
   integer i,j,m,nf,l,k[6],r;
   FILE *fgen;
   printf("ccvd6s: computing change for variable %d...\n",vn);
/*
   check
*/
   if ((vn < 1) || (vn > 4))
      {
         puts("ccvd6s error: vn must be between 1 and 4.");
         printf("actual value: %d\n",vn);
         exit(1);
      }
/*
   we fill up c with the identity change corresponding to the variable vn
*/
   for (i=1; i<=n; i++)
   {
      m=ntph6s(i);
      for (j=0; j<m; j++) c[i][j]=0;
   }
   for (j=0; j<6; j++) k[j]=0;
   k[vn-1]=1;
   j=exll6s(k,1);
   c[1][j]=complex(1,0);
/*
   now we allocate working space. g will be used to contain the
   generating function and u,v will be used as working space to
   hold intermediate values during the computation of the change.
   altough during the computations they will be used to store polynomials
   of different sizes, we will allocate them big enough for the maximum
   size used. this has been done to keep the code simpler and to avoid
   being allocating/freeing memory continuously.
*/
   m=ntph6s(n);
   g=(complex*)malloc(m*sizeof(complex));
   if (g == NULL) {puts("ccvd6s error: out of memory (1)"); exit(1);}
   u=(complex*)malloc(m*sizeof(complex));
   if (u == NULL) {puts("ccvd6s error: out of memory (2)"); exit(1);}
   v=(complex*)malloc(m*sizeof(complex));
   if (v == NULL) {puts("ccvd6s error: out of memory (3)"); exit(1);}
/*
   opening the file with the generating function...
*/
   fgen=fopen(nom,"rb");
   if (fgen == NULL)
      {
         printf("ccvd6s error: can't open the file %s\n",nom);
         puts("(this file must contain the generating function)");
         exit(1);
      }
/*
   the first step (the one that uses the generating function of
   degree 3) can be done more efficiently than the next ones. so,
   it is done separately
*/
   rpb6s(fgen,g,3);
   m=ntph6s(2);
   if ((vn%2) == 0)
      {
         dph6s(g,3,vn-1,u);
         for (j=0; j<m; j++) u[j]=-u[j];
      }
      else
      {
         dph6s(g,3,vn+1,u);
      }
   for (j=0; j<m; j++) c[2][j]=u[j];
   nf=1;
   f=1;
   for (i=3; i<=n; i++)
   {
      ++nf;
      f/=nf;
      papu6s(u,i-1,g,3,v,i,0);
      m=ntph6s(i);
      for (j=0; j<m; j++)
      {
         u[j]=v[j];
         c[i][j]=u[j]*f;
      }
   }
/*
   this is the loop to finish the computation of the change of
   variables. the index r is the degree of the (homogeneous) part of
   the generating function we are actually using.
*/
   for (r=4; r<=n; r++)
   {
      rpb6s(fgen,g,r);
      for (i=n-r+2; i>=1; i--)
      {
         m=ntph6s(i);
         for (j=0; j<m; j++) v[j]=c[i][j];
         f=1;
         nf=0;
         for (l=i+r-2; l<=n; l+=r-2)
         {
            ++nf;
            f/=nf;
            papu6s(v,l-r+2,g,r,u,l,0);
            m=ntph6s(l);
            for (j=0; j<m; j++)
            {
               v[j]=u[j];
               c[l][j]+=u[j]*f;
            }
         }
      }
   }
   free(v);
   free(u);
   free(g);
   fclose(fgen);
   printf("ccvd6s: the change for variable %d has been computed.\n",vn);
   return;
}
void ccvd6p(integer vn, complex **c, integer n, char *nom)
/*
routine to compute the (complex) change of variables corresponding
to variable number vn. it is assumed that the change is handled by
mp6p (this is true for vn=5 and 6 in the RTBP case).

parameters:
vn:  number (5 or 6) pointing to the variable for which the
     transformation should be computed. the numbering of the variables
     is the same for all this work: 5 corresponds to the third position
     and 6 to the third momentum.
c:   space to store the computed change of variables. this space must
     be big enough to store expansions --with the simmetry mp6p-- of
     degree n (output).
n:   degree up to which the change must be computed.
nom: name of the (binary) file containing the generating function of
     the change of variables. it is assumed that the generaing function
     starts from degree 3 up to degree at least n. this file must have
     been written by the routines of io6s.c (input).
*/
{
   complex *g,*u,*v;
   renum f;
   integer i,j,m,nf,l,k[6],r;
   FILE *fgen;
   printf("ccvd6p: computing change for variable %d...\n",vn);
/*
   check
*/
   if ((vn != 5) && (vn != 6))
      {
         puts("ccvd6p error: vn must be 5 or 6.");
         printf("actual value: %d\n",vn);
         exit(1);
      }
/*
   we fill up c with the identity change corresponding to the variable vn
*/
   for (i=1; i<=n; i++)
   {
      m=ntph6p(i);
      for (j=0; j<m; j++) c[i][j]=0;
   }
   for (j=0; j<6; j++) k[j]=0;
   k[vn-1]=1;
   j=exll6p(k,1);
   c[1][j]=complex(1,0);
/*
   now we allocate working space. g will be used to contain the
   generating function and u,v will be used as working space to
   hold intermediate values during the computation of the change.
   altough during the computations they will be used to store polynomials
   of different sizes, we will allocate them big enough for the maximum
   size used. this has been done to keep the code simpler and to avoid
   being allocating/freeing memory continuously.
*/
   m=ntph6s(n);
   g=(complex*)malloc(m*sizeof(complex));
   if (g == NULL) {puts("ccvd6p error: out of memory (1)"); exit(1);}
   m=ntph6p(n);
   u=(complex*)malloc(m*sizeof(complex));
   if (u == NULL) {puts("ccvd6p error: out of memory (2)"); exit(1);}
   v=(complex*)malloc(m*sizeof(complex));
   if (v == NULL) {puts("ccvd6p error: out of memory (3)"); exit(1);}
/*
   opening the file with the generating function...
*/
   fgen=fopen(nom,"rb");
   if (fgen == NULL)
      {
         printf("ccvd6p error: can't open the file %s\n",nom);
         puts("(this file must contain the generating function)");
         exit(1);
      }
/*
   the first step (the one that uses the generating function of
   degree 3) can be done more efficiently than the next ones. so
   it is done separately
*/
   rpb6s(fgen,g,3);
   m=ntph6p(2);
   if ((vn%2) == 0)
      {
         dph6p(g,3,vn-1,u);
         for (j=0; j<m; j++) u[j]=-u[j];
      }
      else
      {
         dph6p(g,3,vn+1,u);
      }
   for (j=0; j<m; j++) c[2][j]=u[j];
   nf=1;
   f=1;
   for (i=3; i<=n; i++)
   {
      ++nf;
      f/=nf;
      papu6sp(u,i-1,g,3,v,i,0);
      m=ntph6p(i);
      for (j=0; j<m; j++)
      {
         u[j]=v[j];
         c[i][j]=u[j]*f;
      }
   }
/*
   this is the loop to finish the computation of the change of
   variables. the index r is the degree of the (homogeneous) part of
   the generating function we are actually using.
*/
   for (r=4; r<=n; r++)
   {
      rpb6s(fgen,g,r);
      for (i=n-r+2; i>=1; i--)
      {
         m=ntph6p(i);
         for (j=0; j<m; j++) v[j]=c[i][j];
         f=1;
         nf=0;
         for (l=i+r-2; l<=n; l+=r-2)
         {
            ++nf;
            f/=nf;
            papu6sp(v,l-r+2,g,r,u,l,0);
            m=ntph6p(l);
            for (j=0; j<m; j++)
            {
               v[j]=u[j];
               c[l][j]+=u[j]*f;
            }
         }
      }
   }
   free(v);
   free(u);
   free(g);
   fclose(fgen);
   printf("ccvd6p: the change for variable %d has been computed.\n",vn);
   return;
}
void ccvi6s(integer vn, complex **c, integer n, char *nom)
/*
routine to compute the inverse (and complex) change of variables
corresponding to variable number vn. it is assumed that the change
is handled by mp6s (this is true for vn=1,2,3 and 4 in the RTBP case).

parameters:
vn:  number (between 1 and 4) pointing to the variable for which the
     transformation should be computed. the numbering of the variables
     is the same for all this work: 1 corresponds to the first position,
     2 to the first momentum, 3 to the second position and 4 to the
     second momentum.
c:   space to store the computed change of variables. this space must
     be big enough to store expansions --with the simmetry mp6s-- of
     degree n (output).
n:   degree up to which the change must be computed.
nom: name of the (binary) file containing the generating function of
     the change of variables. it is assumed that the generating function
     starts from degree 3 up to degree at least n. this file must have
     been written by the routines of io6s.c (input).
*/
{
   complex *g,*u,*v;
   renum f;
   integer i,j,m,nf,l,k[6],r;
   char *flag;
   FILE *fgen;
   printf("ccvi6s: computing inverse change for variable %d...\n",vn);
/*
   check
*/
   if ((vn < 1) || (vn > 4))
      {
         puts("ccvi6s error: vn must be between 1 and 4.");
         printf("actual value: %d\n",vn);
         exit(1);
      }
/*
   we fill up c with the identity change corresponding to the variable vn
*/
   for (i=1; i<=n; i++)
   {
      m=ntph6s(i);
      for (j=0; j<m; j++) c[i][j]=0;
   }
   for (j=0; j<6; j++) k[j]=0;
   k[vn-1]=1;
   j=exll6s(k,1);
   c[1][j]=complex(1,0);
/*
   now we allocate working space. g will be used to contain the
   generating function and u,v will be used as working space to
   hold intermediate values during the computation of the change.
   altough during the computations they will be used to store polynomials
   of different sizes, we will allocate them big enough for the maximum
   size used. this has been done to keep the code simpler and to avoid
   being allocating/freeing memory continuously.
*/
   m=ntph6s(n);
   g=(complex*)malloc(m*sizeof(complex));
   if (g == NULL) {puts("ccvi6s error: out of memory (1)"); exit(1);}
   u=(complex*)malloc(m*sizeof(complex));
   if (u == NULL) {puts("ccvi6s error: out of memory (2)"); exit(1);}
   v=(complex*)malloc(m*sizeof(complex));
   if (v == NULL) {puts("ccvi6s error: out of memory (3)"); exit(1);}
/*
   next we allocate an array of flags, to mark the homogeneous polynomials
   (inside the change of variables) that are zero.
*/
   flag=(char*)malloc((n+1)*sizeof(char));
   if (flag == NULL) {puts("ccvi6s error: out of memory (4)"); exit(1);}
   for (i=0; i<=n; i++) flag[i]='z'; /* 'z' means zero */
   flag[1]='x';                      /* 'x' means nonzero */
/*
   opening the file with the generating function...
*/
   fgen=fopen(nom,"rb");
   if (fgen == NULL)
      {
         printf("ccvi6s error: can't open the file %s\n",nom);
         puts("(this file must contain the generating function)");
         exit(1);
      }
/*
   this is the loop to finish the computation of the change of
   variables. the index r is the degree of the (homogeneous) part of
   the generating function we are actually using.
*/
   for (r=n; r>=3; r--)
   {
      rfgen6s(fgen,g,r);
      m=ntph6s(r);
      for (i=0; i<m; i++) g[i]=-g[i];
      for (i=n-r+2; i>=1; i--)
      {
         if (flag[i] == 'z') continue;
         m=ntph6s(i);
         for (j=0; j<m; j++) v[j]=c[i][j];
         f=1;
         nf=0;
         for (l=i+r-2; l<=n; l+=r-2)
         {
            ++nf;
            f/=nf;
            papu6s(v,l-r+2,g,r,u,l,0);
            flag[l]='x';
            m=ntph6s(l);
            for (j=0; j<m; j++)
            {
               v[j]=u[j];
               c[l][j]+=u[j]*f;
            }
         }
      }
   }
   free(flag);
   free(v);
   free(u);
   free(g);
   fclose(fgen);
   return;
}
void ccvi6p(integer vn, complex **c, integer n, char *nom)
/*
routine to compute the inverse (and complex) change of variables
corresponding to variable number vn. it is assumed that the change
is handled by mp6p (this is true for vn=5 and 6 in the RTBP case).

parameters:
vn:  number (5 or 6) pointing to the variable for which the
     transformation should be computed. the numbering of the variables
     is the same for all this work: 5 corresponds to the third position
     and 6 to the third momentum.
c:   space to store the computed change of variables. this space must
     be big enough to store expansions --with the simmetry mp6p-- of
     degree n (output).
n:   degree up to which the change must be computed.
nom: name of the (binary) file containing the generating function of
     the change of variables. it is assumed that the generaing function
     starts from degree 3 up to degree at least n. this file must have
     been written by the routines of io6s.c (input).
*/
{
   complex *g,*u,*v;
   renum f;
   integer i,j,m,nf,l,k[6],r;
   char *flag;
   FILE *fgen;
   printf("ccvi6p: computing inverse change for variable %d...\n",vn);
/*
   check
*/
   if ((vn != 5) && (vn != 6))
      {
         puts("ccvi6p error: vn must be 5 or 6.");
         printf("actual value: %d\n",vn);
         exit(1);
      }
/*
   we fill up c with the identity change corresponding to the variable vn
*/
   for (i=1; i<=n; i++)
   {
      m=ntph6p(i);
      for (j=0; j<m; j++) c[i][j]=0;
   }
   for (j=0; j<6; j++) k[j]=0;
   k[vn-1]=1;
   j=exll6p(k,1);
   c[1][j]=complex(1,0);
/*
   now we allocate working space. g will be used to contain the
   generating function and u,v will be used as working space to
   hold intermediate values during the computation of the change.
   altough during the computations they will be used to store polynomials
   of different sizes, we will allocate them big enough for the maximum
   size used. this has been done to keep the code simpler and to avoid
   being allocating/freeing memory continuously.
*/
   m=ntph6s(n);
   g=(complex*)malloc(m*sizeof(complex));
   if (g == NULL) {puts("ccvi6p error: out of memory (1)"); exit(1);}
   m=ntph6p(n);
   u=(complex*)malloc(m*sizeof(complex));
   if (u == NULL) {puts("ccvi6p error: out of memory (2)"); exit(1);}
   v=(complex*)malloc(m*sizeof(complex));
   if (v == NULL) {puts("ccvi6p error: out of memory (3)"); exit(1);}
/*
   next we allocate an array of flags, to mark the homogeneous polynomials
   (inside the change of variables) that are zero.
*/
   flag=(char*)malloc((n+1)*sizeof(char));
   if (flag == NULL) {puts("ccvi6p error: out of memory (4)"); exit(1);}
   for (i=0; i<=n; i++) flag[i]='z'; /* 'z' means zero */
   flag[1]='x';                      /* 'x' means nonzero */
/*
   opening the file with the generating function...
*/
   fgen=fopen(nom,"rb");
   if (fgen == NULL)
      {
         printf("ccvi6p error: can't open the file %s\n",nom);
         puts("(this file must contain the generating function)");
         exit(1);
      }
/*
   this is the loop to finish the computation of the change of
   variables. the index r is the degree of the (homogeneous) part of
   the generating function we are actually using.
*/
   for (r=n; r>=3; r--)
   {
      rfgen6s(fgen,g,r);
      m=ntph6s(r);
      for (i=0; i<m; i++) g[i]=-g[i];
      for (i=n-r+2; i>=1; i--)
      {
         if (flag[i] == 'z') continue;
         m=ntph6p(i);
         for (j=0; j<m; j++) v[j]=c[i][j];
         f=1;
         nf=0;
         for (l=i+r-2; l<=n; l+=r-2)
         {
            ++nf;
            f/=nf;
            papu6sp(v,l-r+2,g,r,u,l,0);
            flag[l]='x';
            m=ntph6p(l);
            for (j=0; j<m; j++)
            {
               v[j]=u[j];
               c[l][j]+=u[j]*f;
            }
         }
      }
   }
   free(v);
   free(u);
   free(g);
   fclose(fgen);
   return;
}
